package routes

import (
	"crypto/sha1"
	"errors"
	"fmt"
	"log"
	"net/http"
	"time"

	"github.com/karlseguin/ccache/v2"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/dnsdb-proxy/internal/cache"
	"a.yandex-team.ru/security/dnsdb-proxy/internal/unistat"
	"a.yandex-team.ru/security/dnsdb-proxy/internal/upstream"
)

const (
	cacheTTL     = time.Duration(31 * 24 * time.Hour)
	cacheMaxSize = 100 * 1024
)

var (
	lruCache *ccache.Cache
)

func init() {
	lruCache = ccache.New(ccache.Configure().MaxSize(cacheMaxSize))
}

func newLookupHandler(cacheClient *cache.Client, stat *unistat.Counter) func(c echo.Context) error {
	return func(c echo.Context) error {
		upstreamReq := c.Request().URL.String()
		forceFetch := c.Request().Header.Get("X-Force-Fetch") != ""
		authToken := c.Request().Header.Get("Authorization")
		if authToken == "" {
			return errors.New("empty Authorization header")
		}

		record, cacheStatus, err := fetch(cacheClient, upstreamReq, authToken, forceFetch)
		if err != nil {
			return err
		}

		if record == nil {
			return errors.New("failed to fetch")
		}

		cacheStatus.Stat(stat)
		c.Response().Header().Set("X-Cache-Status", cacheStatus.String())
		return c.Blob(http.StatusOK, "application/json", record.Data)
	}
}

func fetch(cacheClient *cache.Client, upstreamReq string, authToken string, forceFetch bool) (record *cache.Record, cacheStatus cache.Status, err error) {
	cacheKey := calcSha1(upstreamReq)
	if !forceFetch {
		if item := lruCache.Get(cacheKey); item != nil && !item.Expired() && item.Value() != nil {
			record = item.Value().(*cache.Record)
			if !isRecordExpired(record) {
				cacheStatus = cache.Status{Status: cache.StatusHotCache}
				return
			}
		}
	}

	cacheStatus = cache.Status{}
	fetch := true
	var cacheErr error
	record, cacheErr = cacheClient.LookupRecord(cacheKey)
	switch {
	case cacheErr != nil:
		log.Printf("failed to fetch record: %s\n", cacheErr.Error())
		cacheStatus.Status = cache.StatusRefresh
	case record == nil:
		// not found
		cacheStatus.Status = cache.StatusNew
	default:
		cacheStatus.Status = cache.StatusCache
		fetch = forceFetch || isRecordExpired(record)

		if fetch {
			cacheStatus.Status = cache.StatusRefresh
		}
	}

	if fetch {
		log.Printf("fetch from dns db: %s\n", cacheKey)
		upstreamRsp, upstreamErr := upstream.Fetch(upstreamReq, authToken)
		if upstreamErr != nil {
			cacheStatus.Status = cache.StatusUpstreamFail
			if record == nil {
				err = fmt.Errorf("failed to get record from upstream: %s", upstreamErr.Error())
				return
			}
		} else {
			now := time.Now()
			var createdAt time.Time
			if record == nil {
				createdAt = now
			} else {
				createdAt = record.CreatedAt
			}

			record = &cache.Record{
				UpdatedAt: now,
				CreatedAt: createdAt,
				Data:      upstreamRsp,
			}
			cacheErr = cacheClient.InsertRecord(cacheKey, upstreamReq, record)
			if cacheErr != nil {
				log.Printf("failed to save record: %s\n", cacheErr.Error())
			}
		}
	}

	if record != nil {
		lruCache.Set(cacheKey, record, time.Until(record.UpdatedAt.Add(cacheTTL)))
	}
	return
}

func calcSha1(secret string) string {
	return fmt.Sprintf("%x", sha1.Sum([]byte(secret)))
}

func isRecordExpired(report *cache.Record) bool {
	return time.Since(report.UpdatedAt) >= cacheTTL
}
