package unistat

import "sync/atomic"

/*
* documentation for stats semantic: https://wiki.yandex-team.ru/golovan/stat-handle/#defoltnyesigopt-suffiksy
 */
type Counter struct {
	CacheMiss       uint64
	CacheHit        uint64
	CacheInvalidate uint64
	DNSFetch        uint64
	DNSRefresh      uint64
	DNSNew          uint64
	DNSFail         uint64
}

func NewCounter() *Counter {
	return &Counter{}
}

func (c *Counter) AddCacheMiss() {
	atomic.AddUint64(&c.CacheMiss, 1)
}

func (c *Counter) AddCacheHit() {
	atomic.AddUint64(&c.CacheHit, 1)
}

func (c *Counter) AddCacheInvalidate() {
	atomic.AddUint64(&c.CacheInvalidate, 1)
}

func (c *Counter) AddDNSRefresh() {
	atomic.AddUint64(&c.DNSRefresh, 1)
}

func (c *Counter) AddDNSNew() {
	atomic.AddUint64(&c.DNSNew, 1)
}

func (c *Counter) AddDNSFetch() {
	atomic.AddUint64(&c.DNSFetch, 1)
}

func (c *Counter) AddDNSFail() {
	atomic.AddUint64(&c.DNSFail, 1)
}

func (c *Counter) Flush() Counter {
	return Counter{
		CacheMiss:       atomic.SwapUint64(&c.CacheMiss, 0),
		CacheHit:        atomic.SwapUint64(&c.CacheHit, 0),
		CacheInvalidate: atomic.SwapUint64(&c.CacheInvalidate, 0),
		DNSRefresh:      atomic.SwapUint64(&c.DNSRefresh, 0),
		DNSNew:          atomic.SwapUint64(&c.DNSNew, 0),
		DNSFail:         atomic.SwapUint64(&c.DNSFail, 0),
	}
}

func (c *Counter) FlushSignals() [][]interface{} {
	return [][]interface{}{
		{"cache-miss_summ", atomic.SwapUint64(&c.CacheMiss, 0)},
		{"cache-hit_summ", atomic.SwapUint64(&c.CacheHit, 0)},
		{"cache-invalidate_summ", atomic.SwapUint64(&c.CacheInvalidate, 0)},
		{"dns-fetch_summ", atomic.SwapUint64(&c.DNSFetch, 0)},
		{"dns-refresh_summ", atomic.SwapUint64(&c.DNSRefresh, 0)},
		{"dns-new_summ", atomic.SwapUint64(&c.DNSNew, 0)},
		{"dns-fail_summ", atomic.SwapUint64(&c.DNSFail, 0)},
	}
}
