package cleaner

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"time"

	"go.uber.org/zap"
)

type Cleaner struct {
	dir      string
	ttl      time.Duration
	log      *zap.Logger
	done     chan struct{}
	ctx      context.Context
	cancelFn context.CancelFunc
}

func NewCleaner(dir string, opts ...Option) *Cleaner {
	ctx, cancel := context.WithCancel(context.Background())
	out := &Cleaner{
		dir:      dir,
		ttl:      8 * time.Hour,
		log:      zap.NewNop(),
		done:     make(chan struct{}),
		ctx:      ctx,
		cancelFn: cancel,
	}

	for _, opt := range opts {
		opt(out)
	}

	return out
}

func (c *Cleaner) Shutdown(ctx context.Context) error {
	c.cancelFn()

	select {
	case <-ctx.Done():
		return ctx.Err()
	case <-c.done:
		return nil
	}
}

func (c *Cleaner) Start() error {
	defer close(c.done)

	c.doWork()
	ticker := time.NewTicker(10 * time.Minute)
	defer ticker.Stop()

	for {
		select {
		case <-c.ctx.Done():
			return nil
		case <-ticker.C:
			c.doWork()
		}
	}
}

func (c *Cleaner) doWork() {
	c.log.Info("starts snapshot cleanup")
	if err := c.cleanup(); err != nil {
		c.log.Error("cleanup failed", zap.Error(err))
	}
	c.log.Info("snapshot cleared")
}

func (c *Cleaner) cleanup() error {
	files, err := ioutil.ReadDir(c.dir)
	if err != nil {
		return fmt.Errorf("unable to list target directory: %w", err)
	}

	minTime := time.Now().Add(-c.ttl)
	for _, f := range files {
		if !strings.HasPrefix(f.Name(), "ips_") {
			continue
		}

		if f.ModTime().After(minTime) {
			continue
		}

		snapName := f.Name()
		if err := os.Remove(filepath.Join(c.dir, snapName)); err != nil {
			c.log.Error("unable to remove old snapshot", zap.String("name", snapName), zap.Error(err))
			continue
		}

		c.log.Info("removed old snapshot", zap.String("name", snapName), zap.Time("snapshot_time", f.ModTime()), zap.Time("ttl_time", minTime))
	}

	return nil
}
