package commands

import (
	"fmt"
	"time"

	"github.com/spf13/cobra"
	"go.uber.org/zap"

	"a.yandex-team.ru/security/fisthop-collector/internal/parser"
	"a.yandex-team.ru/security/fisthop-collector/internal/synchronizer"
	"a.yandex-team.ru/security/fisthop-collector/internal/zmq"
)

var monitorArgs struct {
	Servers []string
}

var monitorCmd = &cobra.Command{
	Use:          "monitor",
	SilenceUsage: true,
	Short:        "Monitor radius updates",
	RunE: func(_ *cobra.Command, _ []string) error {
		logger := NewLogger()
		gang, err := zmq.NewGang(monitorArgs.Servers, zmq.WithLogger(logger))
		if err != nil {
			logger.Fatal("unable to create zmq gang", zap.Error(err))
		}

		formatTimestamp := func(ts int64) string {
			return fmt.Sprintf("%d: %s", ts, time.Unix(ts, 0))
		}

		for {
			r, err := gang.NextRecord()
			if err != nil {
				logger.Fatal("gang read fail", zap.Error(err))
			}

			switch rr := r.(type) {
			case *parser.AddRecord:
				if rr.RuleKind == parser.RuleKindMAC {
					fmt.Printf("[%s] add mac %s@%s on %s: %s\n", formatTimestamp(rr.Timestamp), rr.Username, parser.EntrypointName(rr.Entrypoint), rr.Hostname, rr.MAC)
				} else {
					fmt.Printf("[%s] add ip %s@%s on %s: %s\n", formatTimestamp(rr.Timestamp), rr.Username, parser.EntrypointName(rr.Entrypoint), rr.Hostname, rr.IP)
				}
			case *parser.DeleteRecord:
				if rr.RuleKind == parser.RuleKindMAC {
					fmt.Printf("[%s] del mac from %s: %s\n", formatTimestamp(rr.Timestamp), rr.Hostname, rr.MAC)
				} else {
					fmt.Printf("[%s] del ip from %s: %s\n", formatTimestamp(rr.Timestamp), rr.Hostname, rr.IP)
				}
			case *parser.RulesetRecord:
				fmt.Printf("[%s] new state received from %s with %d (record)\n", formatTimestamp(rr.Timestamp), rr.Hostname, len(rr.Rules))
				for _, rule := range rr.Rules {
					if time.Now().Sub(time.Unix(rule.Timestamp, 0)) > time.Hour*24*2 {
						fmt.Printf("ignore too old rule for %s: %s\n", rule.Username, time.Now().Sub(time.Unix(rule.Timestamp, 0)))
						continue
					}

					if rule.Kind == parser.RuleKindMAC {
						fmt.Printf("[%s] restore mac %s@%s: %s\n", formatTimestamp(rule.Timestamp), rule.Username, parser.EntrypointName(rule.Entrypoint), rule.MAC)
					} else {
						fmt.Printf("[%s] restore ip %s@%s: %s\n", formatTimestamp(rule.Timestamp), rule.Username, parser.EntrypointName(rule.Entrypoint), rule.IP)
					}
				}
			}
		}
	},
}

func init() {
	flags := monitorCmd.PersistentFlags()
	flags.StringSliceVar(&monitorArgs.Servers, "server", synchronizer.TrustedServers, "firsthop servers to sync with")
}
