package commands

import (
	"context"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/spf13/cobra"
	"go.uber.org/zap"

	"a.yandex-team.ru/security/fisthop-collector/internal/cleaner"
	"a.yandex-team.ru/security/fisthop-collector/internal/synchronizer"
)

var syncArgs struct {
	Servers     []string
	FlushPeriod time.Duration
	OutDir      string
}

var syncCmd = &cobra.Command{
	Use:          "sync",
	SilenceUsage: true,
	Short:        "Starts syncer",
	RunE: func(_ *cobra.Command, _ []string) error {
		logger := NewLogger()
		syncer, err := synchronizer.NewSynchronizer(synchronizer.NewConfig().
			WithLogger(logger).
			WithFlushPeriod(syncArgs.FlushPeriod).
			WithOutDir(syncArgs.OutDir).
			WithTrustedServers(syncArgs.Servers...).
			Build(),
		)
		if err != nil {
			logger.Fatal("unable to create synchronizer", zap.Error(err))
		}

		clean := cleaner.NewCleaner(syncArgs.OutDir, cleaner.WithLogger(logger))

		stopChan := make(chan os.Signal, 1)
		signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

		errChan := make(chan error, 1)
		go func() {
			logger.Info("starting synchronizer")

			if err := syncer.Start(); err != nil {
				errChan <- err
			}
		}()

		go func() {
			logger.Info("starting cleaner")

			if err := clean.Start(); err != nil {
				errChan <- err
			}
		}()

		select {
		case <-stopChan:
			logger.Info("shutting down")
			if err := syncer.Shutdown(context.TODO()); err != nil {
				logger.Error("failed to shutdown synchronizer instance", zap.Error(err))
			}

			if err := clean.Shutdown(context.TODO()); err != nil {
				logger.Error("failed to shutdown cleaner instance", zap.Error(err))
			}
			return err
		case err := <-errChan:
			logger.Error("shit happens", zap.Error(err))
			return err
		}
	},
}

func init() {
	flags := syncCmd.PersistentFlags()
	flags.StringSliceVar(&syncArgs.Servers, "server", synchronizer.TrustedServers, "firsthop servers to sync with")
	flags.StringVar(&syncArgs.OutDir, "out", ".", "output directory for databases")
	flags.DurationVar(&syncArgs.FlushPeriod, "flush-period", synchronizer.DefaultFlushPeriod, "flush period")
}
