package parser

import (
	"bytes"
	"crypto/md5"
	"encoding/binary"
	"fmt"

	"github.com/vmihailenco/msgpack"
)

type Parser struct {
	dec *msgpack.Decoder
}

func NewParser(trusted bool, hostname string) *Parser {
	dec := msgpack.NewDecoder(nil)
	dec.SetDecodeMapFunc(msgDecoder(trusted, hostname))
	return &Parser{
		dec: dec,
	}
}

func (p *Parser) ParseMessage(header, body []byte) (Message, error) {
	h, err := p.parseHeader(header)
	if err != nil {
		return nil, fmt.Errorf("unable to parse header: %w", err)
	}

	out, err := p.parseBody(h, body)
	if err != nil {
		return nil, fmt.Errorf("unable to parse message: %w", err)
	}

	return out, nil
}

func (p *Parser) parseHeader(in []byte) (header, error) {
	var out header
	err := binary.Read(bytes.NewReader(in), binary.BigEndian, &out)
	return out, err
}

func (p *Parser) parseBody(header header, in []byte) (Message, error) {
	if int(header.Size) != len(in) {
		return nil, fmt.Errorf("unexpected message size: %d (header) != %d (actual)", header.Size, len(in))
	}

	if header.Md5Sum != md5.Sum(in) {
		return nil, fmt.Errorf("invalid message checksum: %x (header) != %x (actual)", header.Md5Sum, md5.Sum(in))
	}

	if err := p.dec.Reset(bytes.NewBuffer(in)); err != nil {
		return nil, err
	}

	msg, err := p.dec.DecodeMap()
	if err != nil {
		return nil, fmt.Errorf("unable to parse message: %w", err)
	}

	if msg == nil {
		return nil, nil
	}

	out, ok := msg.(Message)
	if !ok {
		return nil, fmt.Errorf("unexpected msg type: %T", msg)
	}

	return out, nil
}
