package parser

import (
	"fmt"
	"net"
)

// https://bb.yandex-team.ru/projects/NOC/repos/fw-filter/browse/fw-injector.h#68
const (
	EntrypointNone = iota
	EntrypointWired
	EntrypointWireless
	EntrypointVPN
	EntrypointMobile
	EntrypointNAC
)

func EntrypointName(e uint8) string {
	switch e {
	case EntrypointNone:
		return "none"
	case EntrypointWired:
		return "wired"
	case EntrypointWireless:
		return "wireless"
	case EntrypointVPN:
		return "vpn"
	case EntrypointMobile:
		return "mobile"
	case EntrypointNAC:
		return "nac"
	default:
		return fmt.Sprintf("unknown_%d", e)
	}
}

type RuleKind uint8

const (
	RuleKindNone = iota
	RuleKindIPv4
	RuleKindIPv6
	RuleKindMAC
)

func (r RuleKind) String() string {
	switch r {
	case RuleKindNone:
		return "none"
	case RuleKindIPv4:
		return "ipv4"
	case RuleKindIPv6:
		return "ipv6"
	case RuleKindMAC:
		return "mac"
	default:
		return fmt.Sprintf("unknown_%d", r)
	}
}

type Message interface {
	isMessage()
}

type AddRecord struct {
	RuleKind   RuleKind
	Timestamp  int64
	IP         net.IP
	MAC        net.HardwareAddr
	Username   string
	Hostname   string
	Entrypoint uint8
}

func (r AddRecord) isMessage() {}

type DeleteRecord struct {
	RuleKind  RuleKind
	Timestamp int64
	IP        net.IP
	MAC       net.HardwareAddr
	Hostname  string
}

func (r DeleteRecord) isMessage() {}

type UpdatePublishersRecord struct {
	Timestamp int64
	Hostnames []string
}

func (r UpdatePublishersRecord) isMessage() {}

type RulesetRecord struct {
	Timestamp int64
	Hostname  string
	Rules     []Rule
}

func (r RulesetRecord) isMessage() {}

type Rule struct {
	Kind       RuleKind
	Timestamp  int64
	IP         net.IP
	MAC        net.HardwareAddr
	Username   string
	Entrypoint uint8
}

type header struct {
	Magic  [2]byte
	Size   uint32
	Md5Sum [16]byte
}

type binaryData struct {
	Timestamp        int64
	UncompressedSize int
	Data             []byte
	Md5Sum           [16]byte
}
