package stampdb

import (
	"fmt"

	"github.com/mailru/easyjson"
	"github.com/mailru/easyjson/jlexer"
	"github.com/mailru/easyjson/jwriter"

	"a.yandex-team.ru/security/fisthop-collector/internal/parser"
)

//go:generate easyjson

var _ easyjson.Marshaler = (*Via)(nil)
var _ easyjson.Unmarshaler = (*Via)(nil)

type Via uint8

const (
	ViaNone     = Via(parser.EntrypointNone)
	ViaWired    = Via(parser.EntrypointWired)
	ViaWireless = Via(parser.EntrypointWireless)
	ViaVPN      = Via(parser.EntrypointVPN)
	ViaMobile   = Via(parser.EntrypointMobile)
	ViaNAC      = Via(parser.EntrypointNAC)
)

func (v Via) Uint8() uint8 {
	return uint8(v)
}

func (v Via) String() string {
	switch v {
	case ViaNone:
		return "none"
	case ViaWired:
		return "wired"
	case ViaWireless:
		return "wireless"
	case ViaVPN:
		return "vpn"
	case ViaMobile:
		return "mobile"
	case ViaNAC:
		return "nac"
	default:
		return fmt.Sprintf("unknown_%d", v)
	}
}

func (v *Via) FromString(in string) {
	switch in {
	case "wired":
		*v = ViaWired
	case "wireless":
		*v = ViaWireless
	case "vpn":
		*v = ViaVPN
	case "mobile":
		*v = ViaMobile
	case "nac":
		*v = ViaNAC
	default:
		*v = ViaNone
	}
}

func (v *Via) MarshalEasyJSON(w *jwriter.Writer) {
	w.String(v.String())
}

func (v *Via) UnmarshalEasyJSON(w *jlexer.Lexer) {
	v.FromString(w.String())
}

//easyjson:json
type ExportedEntry struct {
	FromTS   int64  `json:"from_ts"`
	ToTS     int64  `json:"to_ts,omitempty"`
	IP       string `json:"ip,omitempty"`
	MAC      string `json:"mac,omitempty"`
	Host     string `json:"host"`
	Username string `json:"username"`
	Via      Via    `json:"via"`
}

type Entry struct {
	Epoch    int64
	AssocTS  int64
	IP       string
	MAC      string
	Username string
	Hostname string
	Via      uint8
}
