package zmq

import (
	"context"
	"errors"
	"fmt"
	"sync"
	"time"

	"go.uber.org/zap"

	"a.yandex-team.ru/security/fisthop-collector/internal/parser"
)

const (
	bufSize = 4096
)

var ErrClosed = errors.New("closed")

type Gang struct {
	subsMu       sync.Mutex
	subsTs       int64
	trustedHosts map[string]bool
	subs         map[string]*subscriber
	msgs         chan parser.Message
	done         chan struct{}
	ctx          context.Context
	cancelFn     context.CancelFunc
	log          *zap.Logger
}

func NewGang(trustedAddrs []string, opts ...GangOption) (*Gang, error) {
	trustedHosts := make(map[string]bool, len(trustedAddrs))
	for _, host := range trustedAddrs {
		trustedHosts[host] = true
	}

	ctx, cancel := context.WithCancel(context.Background())
	out := &Gang{
		trustedHosts: trustedHosts,
		subs:         make(map[string]*subscriber),
		msgs:         make(chan parser.Message, bufSize),
		done:         make(chan struct{}),
		log:          zap.NewNop(),
		ctx:          ctx,
		cancelFn:     cancel,
	}

	for _, opt := range opts {
		opt(out)
	}

	if err := out.refreshSubscribers(time.Now().Unix(), trustedAddrs...); err != nil {
		cancel()
		return nil, fmt.Errorf("unable to refresh subscribers: %w", err)
	}

	return out, nil
}

func (g *Gang) refreshSubscribers(ts int64, hostnames ...string) error {
	g.subsMu.Lock()
	defer g.subsMu.Unlock()

	if g.subsTs == ts {
		return nil
	}

	if ts != 0 && g.subsTs > ts {
		g.log.Warn("skip subscribers update: our newer than remote", zap.Int64("our_ts", g.subsTs), zap.Int64("remote_ts", ts))
		return nil
	}

	if len(hostnames) == 0 {
		return errors.New("no hosts provider")
	}

	// first actualize && connect to the new hosts
	actualHostnames := make(map[string]struct{})
	for _, hostname := range hostnames {
		actualHostnames[hostname] = struct{}{}
		if _, ok := g.subs[hostname]; ok {
			// TODO(buglloc): reconnect to the dead publishers
			continue
		}

		sub := newSubscriber(hostname, g.trustedHosts[hostname], g.log)
		go sub.Subscribe(g.ctx, g.msgs)
		g.subs[hostname] = sub
	}

	// now remove not actual
	for hostname, sub := range g.subs {
		if _, ok := actualHostnames[hostname]; ok {
			continue
		}

		g.log.Info("close subscriber", zap.String("hostname", hostname))
		if err := sub.Close(); err != nil {
			g.log.Warn("unable to close subscriber", zap.String("hostname", hostname), zap.Error(err))
		}

		delete(g.subs, hostname)
	}

	if len(g.subs) == 0 {
		return errors.New("no alive subscribers left")
	}

	return nil
}

func (g *Gang) NextRecord() (parser.Message, error) {
	for {
		select {
		case r := <-g.msgs:
			if r == nil {
				continue
			}

			if updates, ok := r.(*parser.UpdatePublishersRecord); ok {
				g.log.Info("update publishers", zap.Strings("hosts", updates.Hostnames))
				if err := g.refreshSubscribers(updates.Timestamp, updates.Hostnames...); err != nil {
					g.log.Error("unable to update subscribers", zap.Error(err))
				}
				continue
			}

			return r, nil
		case <-g.done:
			return nil, ErrClosed
		}
	}
}

func (g *Gang) Shutdown(_ context.Context) error {
	g.cancelFn()
	for hostname, sub := range g.subs {
		g.log.Info("close subscriber", zap.String("hostname", hostname))
		if err := sub.Close(); err != nil {
			g.log.Warn("unable to close subscriber", zap.String("hostname", hostname), zap.Error(err))
		}

		delete(g.subs, hostname)
	}

	close(g.done)
	return nil
}
