package zmq

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/go-zeromq/zmq4"
	"go.uber.org/zap"

	"a.yandex-team.ru/security/fisthop-collector/internal/parser"
)

const (
	subscribeMagic  = "\x2d\x72"
	connectTimeout  = 1 * time.Second
	reconnectPeriod = 5 * time.Second
)

type subscriber struct {
	hostname string
	trusted  bool
	sock     zmq4.Socket
	log      *zap.Logger
	ctx      context.Context
	cancelFn context.CancelFunc
}

func newSubscriber(hostname string, trusted bool, l *zap.Logger) *subscriber {
	ctx, cancel := context.WithCancel(context.Background())
	return &subscriber{
		hostname: hostname,
		trusted:  trusted,
		sock:     zmq4.NewSub(ctx, zmq4.WithDialerTimeout(connectTimeout)),
		log:      l.With(zap.String("subscriber", hostname)),
		ctx:      ctx,
		cancelFn: cancel,
	}
}

func (s *subscriber) Subscribe(ctx context.Context, out chan<- parser.Message) {
	p := parser.NewParser(s.trusted, s.hostname)
	for {
		func() {
			s.log.Info("connecting")
			defer s.log.Info("disconnected")

			err := s.sock.Dial(fmt.Sprintf("tcp://%s:8548", s.hostname))
			if err != nil {
				s.log.Error("dial failed", zap.Error(err))
				return
			}

			err = s.sock.SetOption(zmq4.OptionSubscribe, subscribeMagic)
			if err != nil {
				s.log.Error("subscription failed", zap.Error(err))
				return
			}

			s.log.Info("connected")
			for {
				msg, err := s.sock.Recv()
				if err != nil {
					if errors.Is(err, context.Canceled) {
						s.log.Info("subscriber closed")
						return
					}

					s.log.Warn("recv failed", zap.Error(err))
					return
				}

				if len(msg.Frames) != 2 {
					s.log.Error("invalid msg: must be 2 fragments", zap.Int("fragments", len(msg.Frames)))
					continue
				}

				parsed, err := p.ParseMessage(msg.Frames[0], msg.Frames[1])
				if err != nil {
					s.log.Error("unable to parse message", zap.Error(err))
					continue
				}

				out <- parsed
			}
		}()

		select {
		case <-ctx.Done():
			return
		case <-s.ctx.Done():
			return
		case <-time.After(reconnectPeriod):
			break
		}
	}
}

func (s *subscriber) Close() error {
	return s.sock.Close()
}
