#include "pkcs11.h"
#include "fuuu-tpm.h"

#include <util/stream/output.h>
#include <util/generic/yexception.h>
#include <library/cpp/logger/global/global.h>

namespace {
    const char *kConfigPath = "/etc/fuuu-tpm.yaml";

    THolder<NFuuuTPM::TFuuuTPM> fuuuTPM = nullptr;

    CK_RV initialize() {
        if (!fuuuTPM) {
            auto cfg = NFuuuTPM::TConfig::FromFile(kConfigPath);
            fuuuTPM.Reset(new NFuuuTPM::TFuuuTPM(cfg));
            InitGlobalLog2Console(cfg.Debug ? TLOG_DEBUG : TLOG_INFO);
        }

        return CKR_OK;
    }

    CK_RV wrapCall(const std::string &name, const std::function<CK_RV()> &f) {
        try {
            DEBUG_LOG << "fuuutpm: called " << name << "()" << Endl;

            return f();
        } catch (const std::exception &e) {
            ERROR_LOG << "fuuutpm: " << name << "() fail: " << e.what() << Endl;
        } catch (...) {
            ERROR_LOG << "fuuutpm: " << name << "() fail: unknown" << Endl;
        }
        return CKR_FUNCTION_FAILED;
    }
}

extern "C" {

CK_RV C_Initialize(CK_VOID_PTR pInitArgs) {
    try {
        CK_RV ret = initialize();
        if (ret != CKR_OK) {
            return ret;
        }
    } catch (const std::exception &e) {
        Cerr << "fuuutpm: initialize fail: " << e.what() << Endl;
        return CKR_FUNCTION_FAILED;
    }

    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->Initialize(pInitArgs);
    });
}

CK_RV C_Finalize(CK_VOID_PTR pReserved) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->Finalize(pReserved);
    });
}

CK_RV C_GetInfo(CK_INFO_PTR pInfo) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetInfo(pInfo);
    });
}

CK_RV C_GetSlotList(CK_BBOOL tokenPresent, CK_SLOT_ID_PTR pSlotList, CK_ULONG_PTR pusCount) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetSlotList(tokenPresent, pSlotList, pusCount);
    });
}

CK_RV C_GetSlotInfo(CK_SLOT_ID slotID, CK_SLOT_INFO_PTR pInfo) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetSlotInfo(slotID, pInfo);
    });
}

CK_RV C_GetTokenInfo(CK_SLOT_ID slotID, CK_TOKEN_INFO_PTR pInfo) {
    return wrapCall(__func__, [&]() -> CK_RV {
        CK_RV ret = fuuuTPM->GetTokenInfo(slotID, pInfo);
        if (ret != CKR_OK) {
            return ret;
        }

        fuuuTPM->MapSavedToken(slotID, pInfo);
        return CKR_OK;
    });
}

CK_RV C_GetMechanismList(CK_SLOT_ID slotID, CK_MECHANISM_TYPE_PTR pMechanismList, CK_ULONG_PTR pusCount) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetMechanismList(slotID, pMechanismList, pusCount);
    });
}

CK_RV C_GetMechanismInfo(CK_SLOT_ID slotID, CK_MECHANISM_TYPE type, CK_MECHANISM_INFO_PTR pInfo) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetMechanismInfo(slotID, type, pInfo);
    });
}

CK_RV C_InitToken(CK_SLOT_ID slotID, CK_CHAR_PTR pPin, CK_ULONG usPinLen, CK_CHAR_PTR pLabel) {
    Y_UNUSED(slotID, pPin, usPinLen, pLabel);
    WARNING_LOG << "fuuutpm: called not implemented C_InitToken()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_InitPIN(CK_SESSION_HANDLE hSession, CK_CHAR_PTR pPin, CK_ULONG usPinLen) {
    Y_UNUSED(hSession, pPin, usPinLen);
    WARNING_LOG << "fuuutpm: called not implemented C_InitPIN()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_SetPIN(CK_SESSION_HANDLE hSession, CK_CHAR_PTR pOldPin, CK_ULONG usOldLen, CK_CHAR_PTR pNewPin,
               CK_ULONG usNewLen) {
    Y_UNUSED(hSession, pOldPin, usOldLen, pNewPin, usNewLen);
    WARNING_LOG << "fuuutpm: called not implemented C_SetPIN()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_OpenSession(CK_SLOT_ID slotID, CK_FLAGS flags, CK_VOID_PTR pApplication, CK_NOTIFY notify,
                    CK_SESSION_HANDLE_PTR phSession) {
    return wrapCall(__func__, [&]() -> CK_RV {
        CK_RV ret = fuuuTPM->OpenSession(slotID, flags, pApplication, notify, phSession);
        if (ret != CKR_OK) {
            return ret;
        }

        const TString &pin = fuuuTPM->UserPin(slotID);
        if (!pin) {
            return ret;
        }

        return fuuuTPM->Login(*phSession, CKU_USER, (CK_UTF8CHAR_PTR) pin.data(), pin.size());
    });
}

CK_RV C_CloseSession(CK_SESSION_HANDLE hSession) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->CloseSession(hSession);
    });
}

CK_RV C_CloseAllSessions(CK_SLOT_ID slotID) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->CloseAllSessions(slotID);
    });
}

CK_RV C_GetSessionInfo(CK_SESSION_HANDLE hSession, CK_SESSION_INFO_PTR pInfo) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetSessionInfo(hSession, pInfo);
    });
}

CK_RV C_GetOperationState(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pOperationState, CK_ULONG_PTR pulOperationStateLen) {
    Y_UNUSED(hSession, pOperationState, pulOperationStateLen);
    WARNING_LOG << "fuuutpm: called not implemented C_GetOperationState()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_SetOperationState(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pOperationState, CK_ULONG ulOperationStateLen,
                          CK_OBJECT_HANDLE hEncryptionKey, CK_OBJECT_HANDLE hAuthenticationKey) {
    Y_UNUSED(hSession, pOperationState, ulOperationStateLen, hEncryptionKey, hAuthenticationKey);
    WARNING_LOG << "fuuutpm: called not implemented C_SetOperationState()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_Login(CK_SESSION_HANDLE hSession, CK_USER_TYPE userType, CK_CHAR_PTR pPin, CK_ULONG usPinLen) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->Login(hSession, userType, pPin, usPinLen);
    });
}

CK_RV C_Logout(CK_SESSION_HANDLE hSession) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->Logout(hSession);
    });
}

CK_RV C_CreateObject(CK_SESSION_HANDLE hSession, CK_ATTRIBUTE_PTR pTemplate, CK_ULONG usCount,
                     CK_OBJECT_HANDLE_PTR phObject) {
    Y_UNUSED(hSession, pTemplate, usCount, phObject);
    WARNING_LOG << "fuuutpm: called not implemented C_CreateObject()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_CopyObject(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObject, CK_ATTRIBUTE_PTR pTemplate, CK_ULONG usCount,
                   CK_OBJECT_HANDLE_PTR phNewObject) {
    Y_UNUSED(hSession, hObject, pTemplate, usCount, phNewObject);
    WARNING_LOG << "fuuutpm: called not implemented C_CopyObject()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DestroyObject(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObject) {
    Y_UNUSED(hSession, hObject);
    WARNING_LOG << "fuuutpm: called not implemented C_DestroyObject()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_GetObjectSize(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObject, CK_ULONG_PTR pusSize) {
    Y_UNUSED(hSession, hObject, pusSize);
    WARNING_LOG << "fuuutpm: called not implemented C_GetObjectSize()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_GetAttributeValue(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObject, CK_ATTRIBUTE_PTR pTemplate,
                          CK_ULONG usCount) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->GetAttributeValue(hSession, hObject, pTemplate, usCount);
    });
}

CK_RV C_SetAttributeValue(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObject, CK_ATTRIBUTE_PTR pTemplate,
                          CK_ULONG usCount) {
    Y_UNUSED(hSession, hObject, pTemplate, usCount);
    WARNING_LOG << "fuuutpm: called not implemented C_SetAttributeValue()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_FindObjectsInit(CK_SESSION_HANDLE hSession, CK_ATTRIBUTE_PTR pTemplate, CK_ULONG usCount) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->FindObjectsInit(hSession, pTemplate, usCount);
    });
}

CK_RV C_FindObjects(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE_PTR phObject, CK_ULONG usMaxObjectCount,
                    CK_ULONG_PTR pusObjectCount) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->FindObjects(hSession, phObject, usMaxObjectCount, pusObjectCount);
    });
}

CK_RV C_FindObjectsFinal(CK_SESSION_HANDLE hSession) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->FindObjectsFinal(hSession);
    });
}

CK_RV C_EncryptInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hKey) {
    Y_UNUSED(hSession, pMechanism, hKey);
    WARNING_LOG << "fuuutpm: called not implemented C_EncryptInit()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_Encrypt(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pData, CK_ULONG usDataLen, CK_BYTE_PTR pEncryptedData,
                CK_ULONG_PTR pusEncryptedDataLen) {
    Y_UNUSED(hSession, pData, usDataLen, pEncryptedData, pusEncryptedDataLen);
    WARNING_LOG << "fuuutpm: called not implemented C_Encrypt()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_EncryptUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pPart, CK_ULONG usPartLen, CK_BYTE_PTR pEncryptedPart,
                      CK_ULONG_PTR pusEncryptedPartLen) {
    Y_UNUSED(hSession, pPart, usPartLen, pEncryptedPart, pusEncryptedPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_EncryptUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_EncryptFinal(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pLastEncryptedPart, CK_ULONG_PTR pusLastEncryptedPartLen) {
    Y_UNUSED(hSession, pLastEncryptedPart, pusLastEncryptedPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_EncryptFinal()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DecryptInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hKey) {
    Y_UNUSED(hSession, pMechanism, hKey);
    WARNING_LOG << "fuuutpm: called not implemented C_DecryptInit()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_Decrypt(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pEncryptedData, CK_ULONG usEncryptedDataLen, CK_BYTE_PTR pData,
                CK_ULONG_PTR pusDataLen) {
    Y_UNUSED(hSession, pEncryptedData, usEncryptedDataLen, pData, pusDataLen);
    WARNING_LOG << "fuuutpm: called not implemented C_Decrypt()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DecryptUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pEncryptedPart, CK_ULONG usEncryptedPartLen,
                      CK_BYTE_PTR pPart, CK_ULONG_PTR pusPartLen) {
    Y_UNUSED(hSession, pEncryptedPart, usEncryptedPartLen, pPart, pusPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DecryptUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DecryptFinal(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pLastPart, CK_ULONG_PTR pusLastPartLen) {
    Y_UNUSED(hSession, pLastPart, pusLastPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DecryptFinal()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DigestInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism) {
    Y_UNUSED(hSession, pMechanism);
    WARNING_LOG << "fuuutpm: called not implemented C_DigestInit()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_Digest(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pData, CK_ULONG usDataLen, CK_BYTE_PTR pDigest,
               CK_ULONG_PTR pusDigestLen) {
    Y_UNUSED(hSession, pData, usDataLen, pDigest, pusDigestLen);
    WARNING_LOG << "fuuutpm: called not implemented C_Digest()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DigestUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pPart, CK_ULONG usPartLen) {
    Y_UNUSED(hSession, pPart, usPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DigestUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DigestKey(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey) {
    Y_UNUSED(hSession, hKey);
    WARNING_LOG << "fuuutpm: called not implemented C_DigestKey()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DigestFinal(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pDigest, CK_ULONG_PTR pusDigestLen) {
    Y_UNUSED(hSession, pDigest, pusDigestLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DigestFinal()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_SignInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hKey) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->SignInit(hSession, pMechanism, hKey);
    });
}

CK_RV C_Sign(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pData, CK_ULONG usDataLen, CK_BYTE_PTR pSignature,
             CK_ULONG_PTR pusSignatureLen) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->Sign(hSession, pData, usDataLen, pSignature, pusSignatureLen);
    });
}

CK_RV C_SignUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pPart, CK_ULONG usPartLen) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->SignUpdate(hSession, pPart, usPartLen);
    });
}

CK_RV C_SignFinal(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pSignature, CK_ULONG_PTR pusSignatureLen) {
    return wrapCall(__func__, [&]() -> CK_RV {
        return fuuuTPM->SignFinal(hSession, pSignature, pusSignatureLen);
    });
}

CK_RV C_SignRecoverInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hKey) {
    Y_UNUSED(hSession, pMechanism, hKey);
    WARNING_LOG << "fuuutpm: called not implemented C_SignRecoverInit()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_SignRecover(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pData, CK_ULONG usDataLen, CK_BYTE_PTR pSignature,
                    CK_ULONG_PTR pusSignatureLen) {
    Y_UNUSED(hSession, pData, usDataLen, pSignature, pusSignatureLen);
    WARNING_LOG << "fuuutpm: called not implemented C_SignRecover()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_VerifyInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hKey) {
    Y_UNUSED(hSession, pMechanism, hKey);
    WARNING_LOG << "fuuutpm: called not implemented C_VerifyInit()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_Verify(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pData, CK_ULONG usDataLen, CK_BYTE_PTR pSignature,
               CK_ULONG usSignatureLen) {
    Y_UNUSED(hSession, pData, usDataLen, pSignature, usSignatureLen);
    WARNING_LOG << "fuuutpm: called not implemented C_Verify()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_VerifyUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pPart, CK_ULONG usPartLen) {
    Y_UNUSED(hSession, pPart, usPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_VerifyUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_VerifyFinal(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pSignature, CK_ULONG usSignatureLen) {
    Y_UNUSED(hSession, pSignature, usSignatureLen);
    WARNING_LOG << "fuuutpm: called not implemented C_VerifyFinal()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_VerifyRecoverInit(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hKey) {
    Y_UNUSED(hSession, pMechanism, hKey);
    WARNING_LOG << "fuuutpm: called not implemented C_VerifyRecoverInit()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_VerifyRecover(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pSignature, CK_ULONG usSignatureLen, CK_BYTE_PTR pData,
                      CK_ULONG_PTR pusDataLen) {
    Y_UNUSED(hSession, pSignature, usSignatureLen, pData, pusDataLen);
    WARNING_LOG << "fuuutpm: called not implemented C_VerifyRecover()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV
C_DigestEncryptUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pPart, CK_ULONG ulPartLen, CK_BYTE_PTR pEncryptedPart,
                      CK_ULONG_PTR pulEncryptedPartLen) {
    Y_UNUSED(hSession, pPart, ulPartLen, pEncryptedPart, pulEncryptedPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DigestEncryptUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DecryptDigestUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pEncryptedPart, CK_ULONG ulEncryptedPartLen,
                            CK_BYTE_PTR pPart, CK_ULONG_PTR pulPartLen) {
    Y_UNUSED(hSession, pEncryptedPart, ulEncryptedPartLen, pPart, pulPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DecryptDigestUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_SignEncryptUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pPart, CK_ULONG ulPartLen, CK_BYTE_PTR pEncryptedPart,
                          CK_ULONG_PTR pulEncryptedPartLen) {
    Y_UNUSED(hSession, pPart, ulPartLen, pEncryptedPart, pulEncryptedPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_SignEncryptUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DecryptVerifyUpdate(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pEncryptedPart, CK_ULONG ulEncryptedPartLen,
                            CK_BYTE_PTR pPart, CK_ULONG_PTR pulPartLen) {
    Y_UNUSED(hSession, pEncryptedPart, ulEncryptedPartLen, pPart, pulPartLen);
    WARNING_LOG << "fuuutpm: called not implemented C_DecryptVerifyUpdate()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV
C_GenerateKey(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_ATTRIBUTE_PTR pTemplate, CK_ULONG usCount,
              CK_OBJECT_HANDLE_PTR phKey) {
    Y_UNUSED(hSession, pMechanism, pTemplate, usCount, phKey);
    WARNING_LOG << "fuuutpm: called not implemented C_GenerateKey()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_GenerateKeyPair(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_ATTRIBUTE_PTR pPublicKeyTemplate,
                        CK_ULONG usPublicKeyAttributeCount, CK_ATTRIBUTE_PTR pPrivateKeyTemplate,
                        CK_ULONG usPrivateKeyAttributeCount, CK_OBJECT_HANDLE_PTR phPrivateKey,
                        CK_OBJECT_HANDLE_PTR phPublicKey) {
    Y_UNUSED(hSession, hSession, pMechanism, pPublicKeyTemplate, usPublicKeyAttributeCount, pPrivateKeyTemplate,
             usPrivateKeyAttributeCount, phPrivateKey, phPublicKey);
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_WrapKey(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hWrappingKey,
                CK_OBJECT_HANDLE hKey, CK_BYTE_PTR pWrappedKey, CK_ULONG_PTR pusWrappedKeyLen) {
    Y_UNUSED(hSession, pMechanism, hWrappingKey, hKey, pWrappedKey, pusWrappedKeyLen);
    WARNING_LOG << "fuuutpm: called not implemented C_WrapKey()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_UnwrapKey(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hUnwrappingKey,
                  CK_BYTE_PTR pWrappedKey, CK_ULONG usWrappedKeyLen, CK_ATTRIBUTE_PTR pTemplate,
                  CK_ULONG usAttributeCount, CK_OBJECT_HANDLE_PTR phKey) {
    Y_UNUSED(hSession, pMechanism, hUnwrappingKey, pWrappedKey, usWrappedKeyLen, pTemplate, usAttributeCount, phKey);
    WARNING_LOG << "fuuutpm: called not implemented C_UnwrapKey()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_DeriveKey(CK_SESSION_HANDLE hSession, CK_MECHANISM_PTR pMechanism, CK_OBJECT_HANDLE hBaseKey,
                  CK_ATTRIBUTE_PTR pTemplate, CK_ULONG usAttributeCount, CK_OBJECT_HANDLE_PTR phKey) {
    Y_UNUSED(hSession, pTemplate, pMechanism, hBaseKey, pTemplate, usAttributeCount, phKey);
    WARNING_LOG << "fuuutpm: called not implemented C_DeriveKey()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_SeedRandom(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pSeed, CK_ULONG usSeedLen) {
    Y_UNUSED(hSession, pSeed, usSeedLen);
    WARNING_LOG << "fuuutpm: called not implemented C_SeedRandom()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_GenerateRandom(CK_SESSION_HANDLE hSession, CK_BYTE_PTR pRandomData, CK_ULONG usRandomLen) {
    Y_UNUSED(hSession, pRandomData, usRandomLen);
    WARNING_LOG << "fuuutpm: called not implemented C_GenerateRandom()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_GetFunctionStatus(CK_SESSION_HANDLE hSession) {
    Y_UNUSED(hSession);
    WARNING_LOG << "fuuutpm: called not implemented C_GetFunctionStatus()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_CancelFunction(CK_SESSION_HANDLE hSession) {
    Y_UNUSED(hSession);
    WARNING_LOG << "fuuutpm: called not implemented C_CancelFunction()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}


CK_RV C_WaitForSlotEvent(CK_FLAGS flags, CK_SLOT_ID_PTR pSlot, CK_VOID_PTR pReserved) {
    Y_UNUSED(flags, pSlot, pReserved);
    WARNING_LOG << "fuuutpm: called not implemented C_WaitForSlotEvent()" << Endl;
    return CKR_FUNCTION_NOT_SUPPORTED;
}

CK_RV C_GetFunctionList(CK_FUNCTION_LIST_PTR_PTR ppFunctionList) {
    static CK_FUNCTION_LIST function_list = {
            .version =  {
                    .major =  0,
                    .minor =  0
            },
            .C_Initialize =  C_Initialize,
            .C_Finalize =  C_Finalize,
            .C_GetInfo =  C_GetInfo,
            .C_GetFunctionList =  C_GetFunctionList,
            .C_GetSlotList =  C_GetSlotList,
            .C_GetSlotInfo =  C_GetSlotInfo,
            .C_GetTokenInfo =  C_GetTokenInfo,
            .C_GetMechanismList =  C_GetMechanismList,
            .C_GetMechanismInfo =  C_GetMechanismInfo,
            .C_InitToken = C_InitToken,
            .C_InitPIN = C_InitPIN,
            .C_SetPIN = C_SetPIN,
            .C_OpenSession = C_OpenSession,
            .C_CloseSession = C_CloseSession,
            .C_CloseAllSessions = C_CloseAllSessions,
            .C_GetSessionInfo = C_GetSessionInfo,
            .C_GetOperationState = C_GetOperationState,
            .C_SetOperationState = C_SetOperationState,
            .C_Login = C_Login,
            .C_Logout = C_Logout,
            .C_CreateObject = C_CreateObject,
            .C_CopyObject = C_CopyObject,
            .C_DestroyObject = C_DestroyObject,
            .C_GetObjectSize = C_GetObjectSize,
            .C_GetAttributeValue = C_GetAttributeValue,
            .C_SetAttributeValue = C_SetAttributeValue,
            .C_FindObjectsInit = C_FindObjectsInit,
            .C_FindObjects = C_FindObjects,
            .C_FindObjectsFinal = C_FindObjectsFinal,
            .C_EncryptInit = C_EncryptInit,
            .C_Encrypt = C_Encrypt,
            .C_EncryptUpdate = C_EncryptUpdate,
            .C_EncryptFinal = C_EncryptFinal,
            .C_DecryptInit = C_DecryptInit,
            .C_Decrypt = C_Decrypt,
            .C_DecryptUpdate = C_DecryptUpdate,
            .C_DecryptFinal = C_DecryptFinal,
            .C_DigestInit = C_DigestInit,
            .C_Digest = C_Digest,
            .C_DigestUpdate = C_DigestUpdate,
            .C_DigestKey = C_DigestKey,
            .C_DigestFinal = C_DigestFinal,
            .C_SignInit = C_SignInit,
            .C_Sign = C_Sign,
            .C_SignUpdate = C_SignUpdate,
            .C_SignFinal = C_SignFinal,
            .C_SignRecoverInit = C_SignRecoverInit,
            .C_SignRecover = C_SignRecover,
            .C_VerifyInit = C_VerifyInit,
            .C_Verify = C_Verify,
            .C_VerifyUpdate = C_VerifyUpdate,
            .C_VerifyFinal = C_VerifyFinal,
            .C_VerifyRecoverInit = C_VerifyRecoverInit,
            .C_VerifyRecover = C_VerifyRecover,
            .C_DigestEncryptUpdate = C_DigestEncryptUpdate,
            .C_DecryptDigestUpdate = C_DecryptDigestUpdate,
            .C_SignEncryptUpdate = C_SignEncryptUpdate,
            .C_DecryptVerifyUpdate = C_DecryptVerifyUpdate,
            .C_GenerateKey = C_GenerateKey,
            .C_GenerateKeyPair = C_GenerateKeyPair,
            .C_WrapKey = C_WrapKey,
            .C_UnwrapKey = C_UnwrapKey,
            .C_DeriveKey = C_DeriveKey,
            .C_SeedRandom = C_SeedRandom,
            .C_GenerateRandom = C_GenerateRandom,
            .C_GetFunctionStatus = C_GetFunctionStatus,
            .C_CancelFunction = C_CancelFunction,
            .C_WaitForSlotEvent = C_WaitForSlotEvent,
    };

    if (ppFunctionList == NULL_PTR) {
        return CKR_ARGUMENTS_BAD;
    }

    *ppFunctionList = &function_list;
    return CKR_OK;
}

}