package isstest

import (
	"net/http"
	"net/http/httptest"
	"sort"
	"strconv"
	"strings"
	"sync"

	"github.com/go-chi/chi/v5"
	"github.com/valyala/fastjson"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/gideon/fake-iss/issres"
)

type Server struct {
	*httptest.Server
	instancesBlackList map[string]struct{}
	instancesMu        sync.RWMutex
	podsBlackList      map[string]struct{}
	podsMu             sync.RWMutex
}

func NewServer(l log.Logger) *Server {
	srv := &Server{
		instancesBlackList: make(map[string]struct{}),
		podsBlackList:      make(map[string]struct{}),
	}

	if l == nil {
		l = &nop.Logger{}
	}

	r := chi.NewRouter()

	r.Get("/instances", func(w http.ResponseWriter, _ *http.Request) {
		srv.instancesMu.RLock()
		defer srv.instancesMu.RUnlock()
		if len(srv.instancesBlackList) == 0 {
			_, _ = w.Write(issres.Instances)
			return
		}

		out, err := fastjson.ParseBytes(issres.Instances)
		if err != nil {
			l.Error("invalid instances data", log.Error(err))
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		instances, err := out.Array()
		if err != nil {
			l.Error("invalid instances data", log.Error(err))
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		var toDelete sort.IntSlice
		for i, v := range instances {
			slotID := string(v.GetStringBytes("slot"))
			_, haveSlot := srv.instancesBlackList[slotID]
			podID := SlotIDToPodID(slotID)
			_, havePod := srv.instancesBlackList[podID]

			if haveSlot || havePod {
				toDelete = append(toDelete, i)
			}
		}

		sort.Sort(sort.Reverse(toDelete))
		for _, idx := range toDelete {
			out.Del(strconv.Itoa(idx))
		}

		_, _ = w.Write(out.MarshalTo(nil))
	})

	r.Get("/pods/info", func(w http.ResponseWriter, _ *http.Request) {
		srv.podsMu.RLock()
		defer srv.podsMu.RUnlock()
		if len(srv.podsBlackList) == 0 {
			_, _ = w.Write(issres.Pods)
			return
		}

		out, err := fastjson.ParseBytes(issres.Pods)
		if err != nil {
			l.Error("invalid post info", log.Error(err))
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		podInfo := out.Get("pod_info")
		if podInfo == nil {
			l.Error("invalid post info", log.Error(err))
			http.Error(w, "invalid resource (no pod_info)", http.StatusInternalServerError)
			return
		}

		pods, err := podInfo.Array()
		if err != nil {
			l.Error("invalid post info", log.Error(err))
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}

		var toDelete sort.IntSlice
		for i, v := range pods {
			podID := v.GetStringBytes("id")
			if _, ok := srv.podsBlackList[string(podID)]; ok {
				toDelete = append(toDelete, i)
			}
		}

		sort.Sort(sort.Reverse(toDelete))
		for _, idx := range toDelete {
			podInfo.Del(strconv.Itoa(idx))
		}

		_, _ = w.Write(out.MarshalTo(nil))
	})

	srv.Server = httptest.NewServer(r)
	return srv
}

func (s *Server) ExcludeSlotID(slotID string) {
	s.instancesMu.Lock()
	defer s.instancesMu.Unlock()
	s.instancesBlackList[slotID] = struct{}{}
}

func (s *Server) ExcludePodID(podID string) {
	s.podsMu.Lock()
	defer s.podsMu.Unlock()
	s.podsBlackList[podID] = struct{}{}
}

func SlotIDToPodID(slotID string) string {
	if idx := strings.IndexByte(slotID, '@'); idx > -1 {
		return slotID[0:idx]
	}

	return slotID
}
