package main

import (
	"fmt"
	"log"
	"os"
	"os/exec"
	"strings"

	"github.com/spf13/pflag"
)

var clangArgs = []string{
	// Code needs to be optimized, otherwise the verifier will often fail
	// to understand it.
	"-O2",
	// Clang defaults to mcpu=probe which checks the kernel that we are
	// compiling on. This isn't appropriate for ahead of time
	// compiled code so force the most compatible version.
	"-mcpu=v1",
	// We always want BTF to be generated, so enforce debug symbols
	"-g",
	// Don't include clang version
	"-fno-ident",
	"-target", "bpf",
	"-D__TARGET_ARCH_x86",
}

func fatalf(format string, args ...interface{}) {
	_, _ = os.Stderr.WriteString(fmt.Sprintf("gideon-cc: "+format+"\n", args...))
	os.Exit(1)
}

func main() {
	input := pflag.String("in", "", "source code")
	out := pflag.String("out", "", "result object path")
	clangPath := pflag.String("clang-path", "", "clang path")
	debugMode := pflag.String("debug-mode", "no", "enable BPF debug output")
	pflag.Parse()

	if *input == "" {
		fatalf("no --input provided")
	}

	if *out == "" {
		fatalf("no --out provided")
	}

	if *clangPath == "" {
		fatalf("no --clang-path provided")
	}

	if *debugMode == "yes" {
		clangArgs = append(clangArgs, "-DBPF_DEBUG")
	}

	clangArgs = append(clangArgs, "-c", *input, "-o", *out)

	clangCmd := exec.Command(*clangPath, clangArgs...)
	clangCmd.Stderr = os.Stderr

	log.Printf("gideon-cc: run clang: %s %s\n", *clangPath, strings.Join(clangArgs, " "))
	if err := clangCmd.Run(); err != nil {
		fatalf("clang failed: %v", err)
	}
}
