package main

import (
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"github.com/spf13/pflag"
)

var (
	clangArgs = []string{
		"-D__KERNEL__",
		"-D__ASM_SYSREG_H",
		"-DKBUILD_MODNAME='\"gideon\"'",
		"-g", "-O2", "-Wall", "-emit-llvm",
		"-Wno-address-of-packed-member",
		"-fno-stack-protector",
	}

	llcArgs = []string{
		"-march=bpf",
		"-filetype=obj",
	}
)

func fatalf(format string, args ...interface{}) {
	_, _ = os.Stderr.WriteString(fmt.Sprintf("gideon-cc: "+format+"\n", args...))
	os.Exit(1)
}

func main() {
	input := pflag.String("in", "", "source code")
	out := pflag.String("out", "", "result object path")
	clangPath := pflag.String("clang-path", "", "clang path")
	llcPath := pflag.String("llc-path", "", "llc path")
	linuxHeadersPath := pflag.String("linux-headers", "", "linux headers path")
	headersPaths := pflag.StringSlice("headers", []string{}, "additional headers path")
	debugMode := pflag.String("debug-mode", "no", "enable BPF debug output")
	pflag.Parse()

	if *input == "" {
		fatalf("no --input provided")
	}

	if *out == "" {
		fatalf("no --out provided")
	}

	if *clangPath == "" {
		fatalf("no --clang-path provided")
	}

	if *llcPath == "" {
		fatalf("no --llc-path provided")
	}

	if *linuxHeadersPath == "" {
		fatalf("no --linux-headers provided")
	}

	tmpPath, err := ioutil.TempDir("", "gideon-cc")
	if err != nil {
		fatalf("unable to create tmp dir: %v", err)
	}

	tmpObjPath := filepath.Join(tmpPath, "obj")
	defer func() {
		_ = os.Remove(tmpPath)
	}()

	if *debugMode == "yes" {
		clangArgs = append(clangArgs, "-DBPF_DEBUG")
	}

	clangIncludes := []string{
		"-isystem", filepath.Join(*linuxHeadersPath, "include"),
		"-isystem", filepath.Join(*linuxHeadersPath, "include", "uapi"),
		"-isystem", filepath.Join(*linuxHeadersPath, "arch", "x86", "include"),
		"-isystem", filepath.Join(*linuxHeadersPath, "arch", "x86", "include", "generated"),
		"-isystem", filepath.Join(*linuxHeadersPath, "arch", "x86", "include", "uapi"),
	}
	if headersPaths != nil {
		for _, h := range *headersPaths {
			clangIncludes = append(clangIncludes, "-isystem", h)
		}
	}

	clangArgs = append(clangArgs, clangIncludes...)
	clangArgs = append(clangArgs, "-c", *input, "-o", tmpObjPath)

	clangCmd := exec.Command(*clangPath, clangArgs...)
	clangCmd.Stderr = os.Stderr

	log.Printf("gideon-cc: run clang: %s %s\n", *clangPath, strings.Join(clangArgs, " "))
	if err := clangCmd.Run(); err != nil {
		fatalf("clang failed: %v", err)
	}

	llcArgs = append(
		llcArgs,
		"-o", *out,
		tmpObjPath,
	)

	llcCmd := exec.Command(*llcPath, llcArgs...)
	llcCmd.Stderr = os.Stderr

	log.Printf("gideon-cc: run llc: %s %s\n", *clangPath, strings.Join(clangArgs, " "))
	if err := llcCmd.Run(); err != nil {
		fatalf("llc failed: %v", err)
	}
}
