package bpf

import (
	"fmt"
)

// EventKind represents event from BPF. Must be in sync with https://a.yandex-team.ru/arc/trunk/arcadia/security/gideon/bpf/prog/events.h
type EventKind uint8

const (
	EventKindUnknown    EventKind = 0
	EventKindMkCgroup   EventKind = 1
	EventKindRmCgroup   EventKind = 2
	EventKindSyscall    EventKind = 3
	EventKindProcExec   EventKind = 4
	EventKindNewSession EventKind = 5
)

func (s EventKind) String() string {
	switch s {
	case EventKindSyscall:
		return "syscall"
	case EventKindMkCgroup:
		return "mk_cgroup"
	case EventKindRmCgroup:
		return "rm_cgroup"
	case EventKindProcExec:
		return "proc_exec"
	case EventKindNewSession:
		return "new_session"
	default:
		return fmt.Sprintf("unknown_%d", s)
	}
}

func EventKindFromUint(u uint8) EventKind {
	switch EventKind(u) {
	case EventKindSyscall:
		return EventKindSyscall
	case EventKindProcExec:
		return EventKindProcExec
	case EventKindMkCgroup:
		return EventKindMkCgroup
	case EventKindRmCgroup:
		return EventKindRmCgroup
	case EventKindNewSession:
		return EventKindNewSession
	default:
		return EventKindUnknown
	}
}

// SyscallKind represents syscall event kind, must in sync with kernel syscall table.
// Hint: https://filippo.io/linux-syscall-table/
type SyscallKind uint16

const (
	SyscallKindUnknown  SyscallKind = 0
	SyscallKindOpen     SyscallKind = 2
	SyscallKindConnect  SyscallKind = 42
	SyscallKindExecVe   SyscallKind = 59
	SyscallKindPtrace   SyscallKind = 101
	SyscallKindOpenAt   SyscallKind = 257
	SyscallKindExecVeAt SyscallKind = 322
)

func (s SyscallKind) Name() string {
	switch s {
	case SyscallKindOpen:
		return "open"
	case SyscallKindConnect:
		return "connect"
	case SyscallKindExecVe:
		return "execve"
	case SyscallKindExecVeAt:
		return "execveat"
	case SyscallKindOpenAt:
		return "openat"
	case SyscallKindPtrace:
		return "ptrace"
	default:
		return fmt.Sprintf("unknown_%d", s)
	}
}

func (s SyscallKind) String() string {
	return s.Name()
}

func SyscallKindUint(u uint16) SyscallKind {
	switch SyscallKind(u) {
	case SyscallKindOpen:
		return SyscallKindOpen
	case SyscallKindConnect:
		return SyscallKindConnect
	case SyscallKindExecVe:
		return SyscallKindExecVe
	case SyscallKindExecVeAt:
		return SyscallKindExecVeAt
	case SyscallKindOpenAt:
		return SyscallKindOpenAt
	case SyscallKindPtrace:
		return SyscallKindPtrace
	default:
		return SyscallKindUnknown
	}
}

type AddressFamilyKind uint16

const (
	AfUnknown AddressFamilyKind = 0x00
	AfInet    AddressFamilyKind = 0x02
	AfInet6   AddressFamilyKind = 0x0A
)

func (s AddressFamilyKind) String() string {
	switch s {
	case AfInet:
		return "AF_INET"
	case AfInet6:
		return "AF_INET6"
	default:
		return fmt.Sprintf("AF_UNKNOWN_%d", s)
	}
}

func AddressFamilyKindUint(u uint16) AddressFamilyKind {
	switch AddressFamilyKind(u) {
	case AfInet:
		return AfInet
	case AfInet6:
		return AfInet6
	default:
		return AfUnknown
	}
}
