package bpf

import (
	"bytes"
	"fmt"

	"github.com/blang/semver/v4"
	"github.com/cilium/ebpf"

	"a.yandex-team.ru/library/go/core/resource"
	"a.yandex-team.ru/security/gideon/gideon/internal/kernel"
)

const (
	moduleResourceKey = "security/gideon/gideon/bpf/gideon_bpf_%s.elf"
)

var availableVersions = []string{
	"4.19.91",
	"4.19.100",
	"4.19.119",
	"4.19.131",
	"4.19.143",
	"5.4.80",
}

func NewCollectionSpec(kernelVer *kernel.Version) (*ebpf.CollectionSpec, string, error) {
	resourceKey, err := chooseResourceKey(kernelVer)
	if err != nil {
		return nil, "", err
	}

	bpfObj := resource.MustGet(resourceKey)
	spec, err := ebpf.LoadCollectionSpecFromReader(bytes.NewReader(bpfObj))
	if err != nil {
		return nil, "", err
	}

	return spec, resourceKey, nil
}

func chooseResourceKey(kernelVer *kernel.Version) (string, error) {
	if kernelVer.Major < 4 {
		return "", fmt.Errorf("unsupported kern version v%s: must be gather than 4.19.x", kernelVer)
	}

	if kernelVer.Major == 4 && kernelVer.Minor < 19 {
		return "", fmt.Errorf("unsupported kern version v%s: must be gather than 4.19.x", kernelVer)
	}

	if kernelVer.Major >= 5 && kernelVer.Minor >= 4 && kernelVer.Patch >= 90 {
		// BTF support enabled for 5.4.90+ kernels:https://st.yandex-team.ru/KERNEL-442#600aae19aa045678d3ff71c8
		return fmt.Sprintf(moduleResourceKey, "btf"), nil
	}

	semCur, err := semver.Parse(kernelVer.String())
	if err != nil {
		return "", fmt.Errorf("can't parse kernel version v%s: %w", kernelVer.String(), err)
	}

	// Choose pre BTF version
	var bestMatch semver.Version
	for _, available := range availableVersions {
		semAvailable := semver.MustParse(available)
		if semAvailable.Major != semCur.Major || semAvailable.Minor != semCur.Minor {
			continue
		}

		if semAvailable.GT(semCur) {
			break
		}
		bestMatch = semAvailable
	}

	if bestMatch.Major == 0 {
		return "", fmt.Errorf("no embedded gideon prog for kernel: v%s", kernelVer)
	}

	return fmt.Sprintf(moduleResourceKey, bestMatch.String()), nil
}
