#ifndef __INTERNAL_H
#define __INTERNAL_H

#include "bpf_helpers.h"
#include "vmlinux/vmlinux.h"
#include "linux_defs.h"

struct perf_event_sample {
    struct perf_event_header header;
    u32 size;
    char data[];
};

#define READ_KERN(ptr)                                                  \
    ({                                                                  \
        typeof(ptr) _val;                                               \
        __builtin_memset((void *)&_val, 0, sizeof(_val));               \
        bpf_core_read((void *)&_val, sizeof(_val), &ptr);               \
        _val;                                                           \
    })

/*
 * Unfortunately the entire perf event length must fit in u16
 */
#define PERF_EVENT_MAX_SIZE (0xffff - sizeof(struct perf_event_sample))

#define RET_SUCCESS 0
#define RET_FAILURE_BUFFER_FULL -1
#define RET_FAILURE_INVALID_USER_MEMORY -2
#define RET_FAILURE_BUG -3
#define RET_IGNORE -4

#define MAX_ARG_SIZE 4096
#define MAX_FAIL_ARG 10
#define SCRATCH_SIZE (1 << 14)
#define SCRATCH_SIZE_MAX (SCRATCH_SIZE - 1)
#define SCRATCH_SIZE_HALF (SCRATCH_SIZE_MAX >> 1)

struct tail_context {
    unsigned long off;
} __attribute__((packed));

struct bpf_per_cpu_state {
    struct tail_context tail_ctx;
} __attribute__((packed));

struct filler_data {
    struct bpf_per_cpu_state *state;
    char *buf;
};

struct bpf_map_def events SEC("maps") = {
    .type = BPF_MAP_TYPE_PERF_EVENT_ARRAY,
    .key_size = sizeof(u32),
    .value_size = sizeof(u32),
    .max_entries = 0,
};

struct bpf_map_def local_state_map SEC("maps") = {
    .type = BPF_MAP_TYPE_PERCPU_ARRAY,
    .key_size = sizeof(u32),
    .value_size = sizeof(struct bpf_per_cpu_state),
    .max_entries = 1,
};

struct bpf_map_def frame_scratch_map SEC("maps") = {
    .type = BPF_MAP_TYPE_PERCPU_ARRAY,
    .key_size = sizeof(u32),
    .value_size = SCRATCH_SIZE,
    // 0xFFFFFFFE mean "possible CPUs"
    .max_entries = 1,
};

// TODO(buglloc): compatible with 5.6+: https://github.com/torvalds/linux/commit/67c0496e87d193b8356d2af49ab95e8a1b954b3c
static __always_inline u64 kernfs_node_id(struct kernfs_node *kn) {
    u64 id;
    BPF_CORE_READ_INTO(&id, kn, id);
    return id;
}

static __always_inline char* get_frame_scratch_area() {
    char* scratch_ptr;
    u32 key = 0;
    scratch_ptr = bpf_map_lookup_elem(&frame_scratch_map, &key);
    if (!scratch_ptr) {
        bpf_printk("frame scratch NULL\n");
    }

    return scratch_ptr;
}

static __always_inline struct bpf_per_cpu_state* get_local_state() {
    u32 key = 0;
    struct bpf_per_cpu_state* state;

    state = bpf_map_lookup_elem(&local_state_map, &key);
    if (!state) {
        bpf_printk("state NULL\n");
    }

    return state;
}

static __always_inline void reset_tail_ctx(struct bpf_per_cpu_state* state) {
    state->tail_ctx.off = 0;
}

static __always_inline u64 get_cgroup_id(struct css_set *cgroups) {
    struct kernfs_node *kn;
    BPF_CORE_READ_INTO(&kn, cgroups, subsys[freezer_cgrp_id], cgroup, kn);
    if (!kn) {
        return 0;
    }

    return kernfs_node_id(kn);
}

static __always_inline struct file *get_task_fd(struct task_struct *task, int fd) {
	struct fdtable *fdt;
	int max_fds;
	struct file **fds;

	BPF_CORE_READ_INTO(&fdt, task, files, fdt);
	if (!fdt) {
	    return NULL;
	}

	BPF_CORE_READ_INTO(&max_fds, fdt, max_fds);
	if (fd >= max_fds) {
	    return NULL;
	}

	BPF_CORE_READ_INTO(&fds, fdt, fd);
	if (!fds) {
	    return NULL;
	}

	return (struct file *)READ_KERN(fds[fd]);
}

static __always_inline u8 read_sock_proto(struct sock *sk) {
    u8 protocol = 0;

    // holly shit, bpf_probe_read can't read bitfields :/
    // TODO(buglloc): do something better

    #if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
        bpf_probe_read(&protocol, 1, (void *)((u64)&sk->sk_gso_max_segs) - 3);
    #elif __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
        bpf_probe_read(&protocol, 1, (void *)((u64)&sk->sk_gso_max_segs) - 1);
    #else
    # error "Fix your compiler's __BYTE_ORDER__?!"
    #endif

    return protocol;
}

static __always_inline bool is_acceptable_sock(struct socket *sock) {
    u16 family;
    u8 proto;
    struct sock *sk;

    BPF_CORE_READ_INTO(&sk, sock, sk);
    if (!sk) {
        return false;
    }

    proto = read_sock_proto(sk);
    if (proto != IPPROTO_TCP) {
        return false;
    }

    BPF_CORE_READ_INTO(&family, sk, sk_family);
    switch (family) {
    case AF_INET:
    case AF_INET6:
        return true;
    default:
        return false;
    }
}

static __always_inline bool is_ipv6_addr_empty(const struct in6_addr *a) {
	const unsigned long *ul = (const unsigned long *)a;

	return (ul[0] | ul[1]) == 0UL;
}

static __always_inline struct socket *get_task_sockfd(struct task_struct *task, int fd) {
	struct file *file;

    if (fd < 0) {
        return NULL;
    }

	file = get_task_fd(task, fd);
	if (!file) {
	    return NULL;
	}

	return BPF_CORE_READ(file, private_data);
}

static __always_inline int bpf_addr_to_kernel(void *uaddr, int ulen, struct __kernel_sockaddr_storage *kaddr) {
	if (ulen <= 0 || (ulen & 0xff) > sizeof(struct __kernel_sockaddr_storage)) {
	    return -EINVAL;
	}

	if (bpf_probe_read(kaddr, ulen & 0xff, uaddr)) {
	    return -EFAULT;
	}

	return 0;
}

static __always_inline const char *get_task_tty_name(struct task_struct *task) {
    return (const char *)BPF_CORE_READ(task, signal, tty, name);
}

#endif
