#ifndef __PROBES_CGROUP_H
#define __PROBES_CGROUP_H

#include "bpf_helpers.h"
#include "internal.h"
#include "filler.h"
#include "settings.h"

/*
linux/include/trace/events/cgroup.h:78
linux/include/trace/events/cgroup.h:85


DEFINE_EVENT(cgroup, cgroup_mkdir,

	TP_PROTO(struct cgroup *cgrp, const char *path),

	TP_ARGS(cgrp, path)
);

DEFINE_EVENT(cgroup, cgroup_rmdir,

	TP_PROTO(struct cgroup *cgrp, const char *path),

	TP_ARGS(cgrp, path)
);
*/
struct cgroup_args {
    unsigned long cgrp;
    unsigned long path;
};


static __always_inline void do_cgroup(enum event_kind kind, struct cgroup_args *args) {
    struct cgroup *cgrp = (struct cgroup *)args->cgrp;
    struct kernfs_node *kn;
    struct filler_data data;
    int ret;

    if (!(u16)(BPF_CORE_READ(cgrp, subtree_control) & (1 << freezer_cgrp_id))) {
        // not a freezer cgroup, just ignore it
        return;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("cgroup: init_filler_data fail: %d\n", ret);
        return;
    }

    reset_tail_ctx(data.state);

    ret = fill_event_header(&data, kind);
    if (ret != RET_SUCCESS) {
        bpf_printk("cgroup: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    BPF_CORE_READ_INTO(&kn, cgrp, kn);
    if (!kn) {
        bpf_printk("cgroup: failed to read cgrp->kn\n");
        goto cleanup;
    }

    ret = fill_data_type(&data, kernfs_node_id(kn), PT_CGRP_ID);
    if (ret != RET_SUCCESS) {
        bpf_printk("cgroup: fill id fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, (unsigned long)args->path, PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("cgroup: fill path fail: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("cgroup: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return;
}

SEC("raw_tracepoint/cgroup_mkdir")
int cgroup_mkdir(struct cgroup_args *args) {
    do_cgroup(EVENT_KIND_MK_CGROUP, args);
    return 0;
}

SEC("raw_tracepoint/cgroup_rmdir")
int cgroup_rmdir(struct cgroup_args *args) {
    do_cgroup(EVENT_KIND_RM_CGROUP, args);
    return 0;
}

#endif
