#ifndef __PROBES_SCHED_H
#define __PROBES_SCHED_H

#include "bpf_helpers.h"
#include "internal.h"
#include "settings.h"
#include "filler.h"
#include "settings.h"
#include "linux_defs.h"

/*
linux/include/trace/events/sched.h:315

TRACE_EVENT(sched_process_exec,

	TP_PROTO(struct task_struct *p, pid_t old_pid,
		 struct linux_binprm *bprm),

	TP_ARGS(p, old_pid, bprm),
*/

struct sched_process_exec_args {
	unsigned long task;
	unsigned long old_pid;
	unsigned long bprm;
};

static __always_inline void do_sched_process_exec(struct sched_process_exec_args *args) {
    struct task_struct *task =  (struct task_struct *)args->task;
    struct linux_binprm *bprm = (struct linux_binprm *)args->bprm;
    unsigned long arg_start;
    unsigned long arg_end;
    unsigned long arg_len;
    struct mm_struct *mm;
    struct filler_data data;
    int ret;

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("sched_process_exec: init_filler_data fail: %d\n", ret);
        return;
    }

    reset_tail_ctx(data.state);

    // we doesn't have session id for this kind of events
    ret = fill_event_header(&data, EVENT_KIND_PROC_EXEC);
    if (ret != RET_SUCCESS) {
        bpf_printk("sched_process_exec: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("sched_process_exec: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    BPF_CORE_READ_INTO(&mm, task, mm);
    if (!mm) {
        bpf_printk("sched_process_exec: no task->mm\n");
        goto cleanup;
    }

    ret = fill_data_type(&data, (unsigned long)BPF_CORE_READ(bprm, filename), PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("sched_process_exec: fill exe fail: %d\n", ret);
        goto cleanup;
    }

    BPF_CORE_READ_INTO(&arg_start, mm, arg_start);
    if (!arg_start) {
        bpf_printk("sched_process_exec: failed to read mm->arg_start\n");
        goto cleanup;
    }

    BPF_CORE_READ_INTO(&arg_end, mm, arg_end);
    if (!arg_end) {
        bpf_printk("sched_process_exec: failed to read mm->arg_end\n");
        goto cleanup;
    }

    arg_len = arg_end - arg_start;
    if (arg_len > 0) {
        ret = fill_data(&data, (unsigned long)arg_start, arg_len - 1, PT_BYTEBUF);
        if (ret != RET_SUCCESS) {
            bpf_printk("sched_process_exec: failed to write proc args (ok ret_code): %d\n", ret);
            goto cleanup;
        }
    } else {
         bpf_printk("sched_process_exec: invalid args_len: %d\n", arg_len);
         goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("sched_process_exec: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return;
}

SEC("raw_tracepoint/sched_process_exec")
int sched_process_exec(struct sched_process_exec_args *args) {
    do_sched_process_exec(args);
    return 0;
}

SEC("raw_tracepoint/sched_process_exec_containers")
int sched_process_exec_containers(struct sched_process_exec_args *args) {
    struct bpf_settings_t *settings;
    struct task_struct *task =  (struct task_struct *)args->task;
    struct css_set *cgroups;
    BPF_CORE_READ_INTO(&cgroups, task, cgroups);
    if (!cgroups) {
       bpf_printk("sched_process_exec_containers: no cgroups\n");
       return 0;
    }

    settings = get_bpf_settings();
    if (!settings || !settings->root_cgroup_id) {
        return 0;
    }

    if (get_cgroup_id(cgroups) == settings->root_cgroup_id) {
        return 0;
    }

    do_sched_process_exec(args);
    return 0;
}

SEC("raw_tracepoint/sched_process_exec_seccomp")
int sched_process_exec_seccomp(struct sched_process_exec_args *args) {
    struct task_struct *task =  (struct task_struct *)args->task;
    int seccomp_mode;

    BPF_CORE_READ_INTO(&seccomp_mode, task, seccomp.mode);
    if (seccomp_mode != SECCOMP_MODE_FILTER) {
        // ignore process w/o seccomp filter
        return 0;
    }

    do_sched_process_exec(args);
    return 0;
}

#endif
