#ifndef __PROBES_SYSCALLS_RAW_H
#define __PROBES_SYSCALLS_RAW_H

#include "bpf_helpers.h"
#include "internal.h"
#include "settings.h"
#include "filler.h"
#include "linux_defs.h"

/*
linux/include/trace/events/syscalls.h:44

TRACE_EVENT_FN(sys_exit,

	TP_PROTO(struct pt_regs *regs, long ret),

	TP_ARGS(regs, ret),
*/

struct sys_exit_args {
	unsigned long regs;
	unsigned long ret;
};

struct bpf_map_def raw_syscall_tail SEC("maps") = {
	.type = BPF_MAP_TYPE_PROG_ARRAY,
	.key_size = sizeof(u32),
	.value_size = sizeof(u32),
	.max_entries = SYSCALL_MAX_N,
};

SEC("raw_tracepoint/sys_exit")
int sys_exit(struct sys_exit_args *args) {
    struct pt_regs *regs = (struct pt_regs *)args->regs;

    int syscall_id;
     BPF_CORE_READ_INTO(&syscall_id, regs, orig_ax);
    if (syscall_id <= 0 || syscall_id >= SYSCALL_MAX_N) {
        return 0;
    }

    bpf_tail_call(args, &raw_syscall_tail, syscall_id);
    return 0;
}

/*
sys_execve(const char __user *filename,
    const char __user *const __user *argv,
    const char __user *const __user *envp);
*/
SEC("raw_tracepoint/syscall/execve")
int trace_sys_execve(struct sys_exit_args *args) {
    long ret_code = args->ret;
    if (ret_code >= 0) {
         // exec call succeeded. sched probe will generate event, no execve needed
        return 0;
    }

    struct pt_regs *regs = (struct pt_regs *)args->regs;
    struct task_struct *task;
    struct filler_data data;
    int ret;

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("trace_sys_execve: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_EXECVE_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_execve: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ret_code, PT_ERRNO);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_execve: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, AT_FDCWD, PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_ARG_0, PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: failed fill filename: %d\n", ret);
        goto cleanup;
    }

    ret = fill_argv_or_env(&data, (char **)SYSCALL_ARG_1);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: failed to write proc args: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}

/*
do_execveat(int fd, struct filename *filename,
	const char __user *const __user *__argv,
	const char __user *const __user *__envp,
	int flags)
*/
SEC("raw_tracepoint/syscall/execveat")
int trace_sys_execveat(struct sys_exit_args *args) {
    long ret_code = args->ret;
    if (ret_code >= 0) {
         // exec call succeeded. sched probe will generate event, no execveat needed
        return 0;
    }

    struct pt_regs *regs = (struct pt_regs *)args->regs;
    struct task_struct *task;
    struct filler_data data;
    int ret;

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execveat: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("trace_sys_execveat: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execveat: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execveat: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_EXECVE_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_execveat: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ret_code, PT_ERRNO);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_execveat: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_ARG_0, PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execveat: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_ARG_1, PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execveat: failed fill filename: %d\n", ret);
        goto cleanup;
    }

    ret = fill_argv_or_env(&data, (char **)SYSCALL_ARG_2);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execveat: failed to write proc args: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_execve: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}


/*
sys_ptrace(long request, long pid, unsigned long addr, unsigned long data);
*/
SEC("raw_tracepoint/syscall/ptrace")
int trace_sys_ptrace(struct sys_exit_args *args) {
    struct pt_regs *regs = (struct pt_regs *)args->regs;
    struct task_struct *task;
    struct filler_data data;
    int ret;
    long ret_code = args->ret;
    s64 ptrace_request = SYSCALL_ARG_0;

    if (ptrace_request != SYS_PTRACE_ATTACH &&
        ptrace_request != SYS_PTRACE_DETACH) {
        // not interesting;
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_ptrace: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("trace_sys_ptrace: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_ptrace: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_ptrace: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_PTRACE, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_ptrace: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ret_code, PT_ERRNO);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_ptrace: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ptrace_request, PT_INT64);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_ptrace: failed fill request: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, (s64)SYSCALL_ARG_1, PT_INT32);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_ptrace: failed to fill pid: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
         bpf_printk("trace_sys_ptrace: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}

/*
connect(int sockfd, const struct sockaddr *addr, socklen_t addrlen);
*/
SEC("raw_tracepoint/syscall/connect")
int trace_sys_connect(struct sys_exit_args *args) {
    struct pt_regs *regs = (struct pt_regs *)args->regs;
    struct task_struct *task;
    struct filler_data data;

    struct __kernel_sockaddr_storage skaddr;
    struct sockaddr *usrskaddr;
    int usrskaddr_len;

    struct socket *sock;

    int ret;
    long ret_code = args->ret;

    if (ret_code < 0 && ret_code != -EINPROGRESS) {
        // ensures we only look for successful (or in progress) connect
        return 0;
    }

    usrskaddr = (struct sockaddr *) SYSCALL_ARG_1;
    usrskaddr_len = (int) SYSCALL_ARG_2;
    if (!usrskaddr || usrskaddr_len <= 0) {
        return 0;
    }

    __builtin_memset(&skaddr, 0, sizeof(skaddr));
    ret = bpf_addr_to_kernel(usrskaddr, usrskaddr_len, &skaddr);
    if (ret) {
        bpf_printk("trace_sys_connect: failed to read user sockaddr: %d\n", ret);
        return 0;
    }

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("trace_sys_connect: no current task\n");
        return 0;
    }

    sock = get_task_sockfd(task, (int)SYSCALL_ARG_0);
    if (!sock) {
        bpf_printk("trace_sys_connect: no socket for fd\n");
        return 0;
    }

    if (!is_acceptable_sock(sock)) {
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_connect: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_connect: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_connect: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_CONNECT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_connect: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_outgoing_sock(&data, sock, &skaddr);
    if (ret == RET_IGNORE) {
        // that's fine
        goto cleanup;
    }

    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_connect: fill_outgoing_sock fail: %d\n", ret);
       goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_ptrace: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}

/*
open(const char *pathname, int flags, mode_t mode)
*/
SEC("raw_tracepoint/syscall/open")
int trace_sys_open(struct sys_exit_args *args) {
    struct pt_regs *regs = (struct pt_regs *)args->regs;
    struct task_struct *task;
    struct filler_data data;
    int ret;
    long ret_code = args->ret;
    int open_flags = SYSCALL_ARG_1;

    if ((open_flags & O_WRONLY) == 0 &&
        (open_flags & O_RDWR) == 0) {
        // not interesting;
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("trace_sys_open: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_OPEN_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_open: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ret_code, PT_FD);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_open: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, AT_FDCWD, PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_ARG_0, PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: failed fill pathname: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, open_flags, PT_INT32);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: failed to flags: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_open: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}

/*
openat(int dirfd, const char *pathname, int flags, mode_t mode);
*/
SEC("raw_tracepoint/syscall/openat")
int trace_sys_openat(struct sys_exit_args *args) {
    struct pt_regs *regs = (struct pt_regs *)args->regs;
    struct task_struct *task;
    struct filler_data data;
    int ret;
    long ret_code = args->ret;
    int open_flags = SYSCALL_ARG_2;

    if ((open_flags & O_WRONLY) == 0 &&
        (open_flags & O_RDWR) == 0) {
        // not interesting;
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("trace_sys_openat: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_OPEN_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_openat: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ret_code, PT_FD);
    if (ret != RET_SUCCESS) {
       bpf_printk("trace_sys_openat: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_ARG_0, PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_ARG_1, PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: failed fill pathname: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, open_flags, PT_INT32);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: failed to flags: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(args, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("trace_sys_openat: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}
#endif
