#ifndef __PROBES_SYSCALLS_SECCOMP_H
#define __PROBES_SYSCALLS_SECCOMP_H

#include "bpf_helpers.h"
#include "internal.h"
#include "settings.h"
#include "filler.h"
#include "linux_defs.h"

struct bpf_map_def seccomp_syscall_tail SEC("maps") = {
	.type = BPF_MAP_TYPE_PROG_ARRAY,
	.key_size = sizeof(u32),
	.value_size = sizeof(u32),
	.max_entries = SYSCALL_MAX_N,
};

/*
__seccomp_filter(
    int this_syscall,
    const struct seccomp_data *sd,
	const bool recheck_after_trace);
*/
SEC("kprobe/__seccomp_filter")
int kprobe__seccomp_filter(struct pt_regs *ctx) {
    int syscall_id = PT_REGS_PARM1_CORE(ctx);
    if (syscall_id <= 0 || syscall_id >= SYSCALL_MAX_N) {
        return 0;
    }

    bpf_tail_call(ctx, &seccomp_syscall_tail, syscall_id);
    return 0;
}


/*
sys_execve(const char __user *filename,
    const char __user *const __user *argv,
    const char __user *const __user *envp);
*/
SEC("kprobe/syscall/execve")
int kprobe_sys_execve(struct pt_regs *ctx) {
    struct task_struct *task;
    struct seccomp_data *sd;
    struct filler_data data;
    int ret;

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    sd = (struct seccomp_data *) PT_REGS_PARM2_CORE(ctx);
    if (!sd) {
       bpf_printk("kprobe_sys_execve: unstash_seccomp_data fail\n");
       goto cleanup;
    }

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_sys_execve: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_EXECVE_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_execve: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    // we don't known actual retcode
    ret = fill_data_type(&data, 0, PT_ERRNO);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_execve: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, AT_FDCWD, PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, BPF_CORE_READ(sd, args[0]), PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: failed fill filename: %d\n", ret);
        goto cleanup;
    }

    ret = fill_argv_or_env(&data, (char **)BPF_CORE_READ(sd, args[1]));
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: failed to write proc args: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execve: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}


/*
do_execveat(int fd, struct filename *filename,
	const char __user *const __user *__argv,
	const char __user *const __user *__envp,
	int flags)
*/
SEC("kprobe/syscall/execveat")
int kprobe_sys_execveat(struct pt_regs *ctx) {
    struct task_struct *task;
    struct seccomp_data *sd;
    struct filler_data data;
    int ret;

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    sd = (struct seccomp_data *) PT_REGS_PARM2_CORE(ctx);
    if (!sd) {
       bpf_printk("kprobe_sys_execveat: unstash_seccomp_data fail\n");
       goto cleanup;
    }

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_sys_execveat: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_EXECVE_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_execveat: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    // we don't known actual retcode
    ret = fill_data_type(&data, 0, PT_ERRNO);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_execveat: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, BPF_CORE_READ(sd, args[0]), PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, BPF_CORE_READ(sd, args[1]), PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: failed fill filename: %d\n", ret);
        goto cleanup;
    }

    ret = fill_argv_or_env(&data, (char **)BPF_CORE_READ(sd, args[2]));
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: failed to write proc args: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_execveat: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}


/*
sys_ptrace(long request, long pid, unsigned long addr, unsigned long data);
*/
SEC("kprobe/syscall/ptrace")
int kprobe_sys_ptrace(struct pt_regs *ctx) {
    struct task_struct *task;
    struct seccomp_data *sd;
    struct filler_data data;
    int ret;
    s64 ptrace_request;

    sd = (struct seccomp_data *) PT_REGS_PARM2_CORE(ctx);
    if (!sd) {
       bpf_printk("kprobe_sys_ptrace: unstash_seccomp_data fail\n");
       goto cleanup;
    }

    ptrace_request = BPF_CORE_READ(sd, args[0]);
    if (ptrace_request != SYS_PTRACE_ATTACH &&
        ptrace_request != SYS_PTRACE_DETACH) {
        // not interesting;
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_sys_ptrace: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_PTRACE, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_ptrace: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    // we don't known real retcode
    ret = fill_data_type(&data, 0, PT_ERRNO);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_ptrace: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, ptrace_request, PT_INT64);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: failed fill request: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, (s64)BPF_CORE_READ(sd, args[1]), PT_INT32);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: failed to fill pid: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}


/*
connect(int sockfd, const struct sockaddr *addr, socklen_t addrlen);
*/
SEC("kprobe/syscall/connect")
int kprobe_sys_connect(struct pt_regs *ctx) {
    struct task_struct *task;
    struct seccomp_data *sd;
    struct filler_data data;
    int ret;

    struct __kernel_sockaddr_storage skaddr;
    struct sockaddr *usrskaddr;
    int usrskaddr_len;

    struct socket *sock;

    sd = (struct seccomp_data *) PT_REGS_PARM2_CORE(ctx);
    if (!sd) {
       bpf_printk("kprobe_sys_connect: unstash_seccomp_data fail\n");
       goto cleanup;
    }

    usrskaddr = (struct sockaddr *) BPF_CORE_READ(sd, args[1]);
    usrskaddr_len = (int) BPF_CORE_READ(sd, args[2]);
    if (!usrskaddr || usrskaddr_len <= 0) {
        bpf_printk("kprobe_sys_connect: no usrskaddr\n");
        goto cleanup;
    }

    __builtin_memset(&skaddr, 0, sizeof(skaddr));
    ret = bpf_addr_to_kernel(usrskaddr, usrskaddr_len, &skaddr);
    if (ret) {
        bpf_printk("kprobe_sys_connect: failed to read user sockaddr: %d\n", ret);
        goto cleanup;
    }

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_sys_connect: no current task\n");
        goto cleanup;
    }

    sock = get_task_sockfd(task, (int) BPF_CORE_READ(sd, args[0]));
    if (!sock) {
        bpf_printk("kprobe_sys_connect: no socket for fd\n");
        goto cleanup;
    }

    if (!is_acceptable_sock(sock)) {
        goto cleanup;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_connect: init_filler_data fail: %d\n", ret);
        goto cleanup;
    }

    reset_tail_ctx(data.state);

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_connect: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_connect: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_CONNECT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_connect: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_outgoing_sock(&data, sock, &skaddr);
    if (ret == RET_IGNORE) {
        // that's fine
        bpf_printk("kprobe_sys_connect: socket ignored\n");
        goto cleanup;
    }

    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_connect: fill_outgoing_sock fail: %d\n", ret);
       goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_ptrace: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}

/*
open(const char *pathname, int flags, mode_t mode)
*/
SEC("kprobe/syscall/open")
int kprobe_sys_open(struct pt_regs *ctx) {
    struct task_struct *task;
    struct seccomp_data *sd;
    struct filler_data data;
    int ret;
    int open_flags;

    sd = (struct seccomp_data *) PT_REGS_PARM2_CORE(ctx);
    if (!sd) {
       bpf_printk("kprobe_sys_open: unstash_seccomp_data fail\n");
       return 0;
    }

    open_flags = BPF_CORE_READ(sd, args[1]);
    if ((open_flags & O_WRONLY) == 0 &&
        (open_flags & O_RDWR) == 0) {
        // not interesting;
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_sys_open: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_OPEN_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_open: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    // we don't known actual retcode
    ret = fill_data_type(&data, 0, PT_FD);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_open: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, AT_FDCWD, PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, BPF_CORE_READ(sd, args[0]), PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: failed fill pathname: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, open_flags, PT_INT32);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: failed fill filename: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_open: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}

/*
openat(int dirfd, const char *pathname, int flags, mode_t mode);
*/
SEC("kprobe/syscall/openat")
int kprobe_sys_openat(struct pt_regs *ctx) {
    struct task_struct *task;
    struct seccomp_data *sd;
    struct filler_data data;
    int ret;
    int open_flags;

    sd = (struct seccomp_data *) PT_REGS_PARM2_CORE(ctx);
    if (!sd) {
       bpf_printk("kprobe_sys_openat: unstash_seccomp_data fail\n");
       return 0;
    }

    open_flags = BPF_CORE_READ(sd, args[2]);
    if ((open_flags & O_WRONLY) == 0 &&
        (open_flags & O_RDWR) == 0) {
        // not interesting;
        return 0;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: init_filler_data fail: %d\n", ret);
        return 0;
    }

    reset_tail_ctx(data.state);

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_sys_openat: no current task\n");
        goto cleanup;
    }

    ret = fill_event_header(&data, EVENT_KIND_SYSCALL);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, SYSCALL_OPEN_AT, PT_UINT16);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_openat: syscall kind fail: %d\n", ret);
       goto cleanup;
    }

    // we don't known actual retcode
    ret = fill_data_type(&data, 0, PT_FD);
    if (ret != RET_SUCCESS) {
       bpf_printk("kprobe_sys_openat: fill ret val fail: %d\n", ret);
       goto cleanup;
    }

    ret = fill_data_type(&data, BPF_CORE_READ(sd, args[0]), PT_FD);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: failed fill fd: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, BPF_CORE_READ(sd, args[1]), PT_CHARBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: failed fill pathname: %d\n", ret);
        goto cleanup;
    }

    ret = fill_data_type(&data, open_flags, PT_INT32);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: failed fill filename: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_sys_openat: flush_ring fail: %d\n", ret);
        goto cleanup;
    }

cleanup:
    return 0;
}
#endif
