#ifndef __FILLER_H
#define __FILLER_H

#include "bpf_helpers.h"
#include "internal.h"
#include "settings.h"

#define EOF 0xEFBEADDE

enum filler_param_type {
    PT_NONE = 0,
    PT_INT8 = 1,
    PT_INT16 = 2,
    PT_INT32 = 3,
    PT_INT64 = 4,
    PT_UINT8 = 5,
    PT_UINT16 = 6,
    PT_UINT32 = 7,
    PT_UINT64 = 8,
    PT_CHARBUF = 9,     /* A printable buffer of bytes, NULL terminated */
    PT_BYTEBUF = 10,    /* A raw buffer of bytes not suitable for printing */
    PT_ERRNO = 11,      /* this is an INT64, but will be interpreted as an error code */
    PT_SOCKADDR = 12,   /* A sockaddr structure, 1byte family + data */
    PT_SOCKTUPLE = 13,  /* A sockaddr tuple,1byte family + 12byte data + 12byte data */
    PT_FD = 14,         /* An fd, 64bit */
    PT_PID = 15,        /* A pid/tid, 64bit */
    PT_FDLIST = 16,     /* A list of fds, 16bit count + count * (64bit fd + 16bit flags) */
    PT_FSPATH = 17,     /* A string containing a relative or absolute file system
                         path, null terminated */
    PT_SYSCALLID = 18,  /* A 16bit system call ID. Can be used as a key for the
                         g_syscall_info_table table. */
    PT_SIGTYPE = 19,    /* An 8bit signal number */
    PT_RELTIME = 20,    /* A relative time. Seconds * 10^9  + nanoseconds. 64bit. */
    PT_ABSTIME = 21,    /* An absolute time interval. Seconds from epoch * 10^9  +
                         nanoseconds. 64bit. */
    PT_PORT = 22,       /* A TCP/UDP prt. 2 bytes. */
    PT_L4PROTO = 23,    /* A 1 byte IP protocol type. */
    PT_SOCKFAMILY = 24, /* A 1 byte socket family. */
    PT_BOOL = 25,       /* A boolean value, 4 bytes. */
    PT_IPV4ADDR = 26,   /* A 4 byte raw IPv4 address. */
    PT_DYN = 27,        /* Type can vary depending on the context. Used for filter fields
                         like evt.rawarg. */
    PT_FLAGS8 = 28,     /* this is an UINT8, but will be interpreted as 8 bit flags. */
    PT_FLAGS16 = 29,    /* this is an UINT16, but will be interpreted as 16 bit flags. */
    PT_FLAGS32 = 30,    /* this is an UINT32, but will be interpreted as 32 bit flags. */
    PT_UID = 31,        /* this is an UINT32, MAX_UINT32 will be interpreted as no value. */
    PT_GID = 32,        /* this is an UINT32, MAX_UINT32 will be interpreted as no value. */
    PT_DOUBLE = 33,     /* this is a double precision floating point number. */
    PT_SIGSET = 34,     /* sigset_t. I only store the lower UINT32 of it */
    PT_CHARBUFARRAY = 35,       /* Pointer to an array of strings, exported by the user
                                 events decoder. 64bit. For internal use only. */
    PT_CHARBUF_PAIR_ARRAY = 36, /* Pointer to an array of string pairs, exported by the user
                                 events decoder. 64bit. For internal use only. */
    PT_IPV4NET = 37,            /* An IPv4 network. */
    PT_IPV6ADDR = 38,           /* A 16 byte raw IPv6 address. */
    PT_IPV6NET = 39,            /* An IPv6 network. */
    PT_IPADDR = 40, /* Either an IPv4 or IPv6 address. The length indicates which
                     one it is. */
    PT_IPNET = 41,  /* Either an IPv4 or IPv6 network. The length indicates which
                     one it is. */
    PT_MODE = 42,   /* a 32 bit bitmask to represent file modes. */
    PT_LEN = 43,
    PT_CGRP_ID = 44, /* a cgroup id */
    PT_MAX = 45,     /* array size */
};


static __always_inline int init_filler_data(struct filler_data *data) {
    data->buf = get_frame_scratch_area();
    if (!data->buf) {
        return RET_FAILURE_BUG;
    }

    data->state = get_local_state();
    if (!data->state) {
        return RET_FAILURE_BUG;
    }

    return RET_SUCCESS;
}

static __always_inline int fill_data(struct filler_data *data, unsigned long val, unsigned long val_len, enum filler_param_type type) {
    unsigned int len = 0;

    if (data->state->tail_ctx.off > SCRATCH_SIZE_HALF - sizeof(u32)) {
        return RET_FAILURE_BUFFER_FULL;
    }

    switch (type) {
    case PT_BYTEBUF: {
        if (val_len >= MAX_ARG_SIZE) {
            val_len = MAX_ARG_SIZE - sizeof(u16);
        }

        len = sizeof(u16);
        *((u16*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = (u16)val_len;

        if (val_len > 0) {
            int res = bpf_probe_read(&data->buf[(data->state->tail_ctx.off + len) & SCRATCH_SIZE_HALF], val_len & SCRATCH_SIZE_HALF, (void*)val);
            if (res != RET_SUCCESS) {
                return RET_FAILURE_INVALID_USER_MEMORY;
            }
            len += val_len;
      }
      break;
    }
    case PT_CHARBUF: {
        if (val_len >= MAX_ARG_SIZE || val_len <= 0) {
            val_len = MAX_ARG_SIZE - sizeof(u16);
        }

        int res = bpf_probe_read_str(&data->buf[(data->state->tail_ctx.off + sizeof(u16)) & SCRATCH_SIZE_HALF], val_len & SCRATCH_SIZE_HALF, (const void*)val);
        if (res < 0) {
            return RET_FAILURE_INVALID_USER_MEMORY;
        }

        if (res > 0) {
            // trim null byte
            res--;
            len = res + sizeof(u16);
            *((u16*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = (u16)res;
        }

        break;
    }
    case PT_FLAGS8:
    case PT_UINT8:
    case PT_SIGTYPE:
        *((u8*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(u8);
        break;
    case PT_FLAGS16:
    case PT_UINT16:
    case PT_SYSCALLID:
        *((u16*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(u16);
        break;
    case PT_FLAGS32:
    case PT_MODE:
    case PT_UINT32:
    case PT_UID:
    case PT_GID:
    case PT_SIGSET:
    case PT_LEN:
        *((u32*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(u32);
        break;
    case PT_RELTIME:
    case PT_ABSTIME:
    case PT_CGRP_ID:
    case PT_UINT64:
        *((u64*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(u64);
        break;
    case PT_INT8:
        *((s8*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(s8);
        break;
    case PT_INT16:
        *((s16*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(s16);
        break;
    case PT_INT32:
        *((s32*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(s32);
        break;
    case PT_INT64:
    case PT_ERRNO:
    case PT_FD:
    case PT_PID:
        *((s64*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = val;
        len = sizeof(s64);
        break;
    default: {
        bpf_printk("unhandled type in fill_data\n");
        return RET_FAILURE_BUG;
    }
    }

    if (len == 0) {
        return RET_SUCCESS;
    }

    if (len > MAX_ARG_SIZE) {
        return RET_FAILURE_BUFFER_FULL;
    }

    data->state->tail_ctx.off += len;

    return RET_SUCCESS;
}

static __always_inline int fill_data_type(struct filler_data *data, unsigned long val, enum filler_param_type type) {
return  fill_data(data, val, 0, type);
}

static __always_inline int fill_event_header(struct filler_data *data, enum event_kind kind) {
    struct event_header_t *h = (struct event_header_t *)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF];
    h->kind = kind;
    h->ts = bpf_ktime_get_ns();
    data->state->tail_ctx.off += sizeof(struct event_header_t);
    return RET_SUCCESS;
}

static __always_inline int fill_eof(struct filler_data *data) {
    return fill_data_type(data, (unsigned long)EOF, PT_LEN);
}


static __always_inline int fill_procinfo(struct filler_data *data, struct task_struct *task) {
    struct task_struct *parent;
    struct css_set *cgroups;

    cgroups = BPF_LOAD(task->cgroups);
    if (!cgroups) {
       bpf_printk("fill_procinfo: no cgroups\n");
       return RET_FAILURE_BUG;
    }

    parent = BPF_LOAD(task->real_parent);
    if (!parent) {
       bpf_printk("fill_procinfo: no parent\n");
       return RET_FAILURE_BUG;
    }

	struct proc_info_t *pi = (struct proc_info_t *)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF];

	pi->cgid = get_cgroup_id(cgroups);
	pi->uid = bpf_get_current_uid_gid() & 0xffffffff;
	pi->sessid = BPF_LOAD(task->sessionid);
	pi->pid = bpf_get_current_pid_tgid() >> 32;
	pi->ppid = BPF_LOAD(parent->pid);
	bpf_probe_read(&pi->comm, sizeof(pi->comm), BPF_LOAD(task->comm));
	bpf_probe_read(&pi->parent_comm, sizeof(pi->parent_comm), BPF_LOAD(parent->comm));

	data->state->tail_ctx.off += sizeof(struct proc_info_t);

    return RET_SUCCESS;
}

static __always_inline int fill_argv_or_env(struct filler_data *data, char **argv) {
	char *arg;
	int off;
	int len;
	int j;
	long args_len = 0;

	off = data->state->tail_ctx.off + sizeof(u16);

    if (!argv) {
        goto out;
    }

	#pragma unroll
	for (j = 0; j < MAX_FAIL_ARG; ++j) {
	    arg = 0;
	    bpf_probe_read(&arg, sizeof(arg), &argv[j]);
		if (!arg) {
		    goto out;
		}

		if (off > SCRATCH_SIZE_HALF) {
		    return RET_FAILURE_BUFFER_FULL;
		}

		len = bpf_probe_read_str(&data->buf[off & SCRATCH_SIZE_HALF], SCRATCH_SIZE_HALF, arg);
		if (len < 0) {
		    return len;
		}

		args_len += len;
		off += len;

		if (args_len > MAX_ARG_SIZE) {
			break;
		}
	}

    // added ellipsis
    const char ellipsis[] = "...";
    bpf_probe_read_str(&data->buf[off & SCRATCH_SIZE_HALF], sizeof(ellipsis), ellipsis);
    // null byte padding
    args_len += 4;
    off += 4;

out:
    if (args_len > 0) {
        // drop last null-byte
        args_len--;
        off--;
    }

    *((u16*)&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF]) = (u16)args_len;
    data->state->tail_ctx.off = off;
	return RET_SUCCESS;
}

static __always_inline int fill_outgoing_sock(struct filler_data *data, struct socket *sock, struct sockaddr_storage *skaddr) {
    u16 family;
    struct sock *sk;
    struct inet_sock *inet;
    int ret;

    sk = BPF_LOAD(sock->sk);
    if (!sk) {
        return RET_FAILURE_BUG;
    }

    family = BPF_LOAD(sk->sk_family);
    ret = fill_data_type(data, family, PT_UINT16);
    if (ret != RET_SUCCESS) {
       return RET_FAILURE_BUG;
    }

    inet = (struct inet_sock *)sk;
    switch (family) {
    case AF_INET: {
        struct sockaddr_in *skaddr_in = (struct sockaddr_in *)skaddr;

        u32 saddr = BPF_LOAD(inet->inet_rcv_saddr);
        u16 sport = ntohs(BPF_LOAD(inet->inet_sport));
        u32 daddr = skaddr_in->sin_addr.s_addr;
        u16 dport = ntohs(skaddr_in->sin_port);

        if (!daddr) {
            return RET_IGNORE;
        }

        __builtin_memcpy(&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF], &saddr, 4);
        __builtin_memcpy(&data->buf[(data->state->tail_ctx.off + 4) & SCRATCH_SIZE_HALF], &sport, 2);
        __builtin_memcpy(&data->buf[(data->state->tail_ctx.off + 6) & SCRATCH_SIZE_HALF], &daddr, 4);
        __builtin_memcpy(&data->buf[(data->state->tail_ctx.off + 10) & SCRATCH_SIZE_HALF], &dport, 2);

        data->state->tail_ctx.off += 4 + 2 + 4 + 2;
        break;
    }
    case AF_INET6: {
        struct sockaddr_in6 *skaddr_in6 = (struct sockaddr_in6 *)skaddr;

        struct in6_addr saddr = BPF_LOAD(sk->sk_v6_rcv_saddr);
        u16 sport = ntohs(BPF_LOAD(inet->inet_sport));
        u8 *daddr = skaddr_in6->sin6_addr.s6_addr;
        u16 dport = ntohs(skaddr_in6->sin6_port);

        if (!daddr) {
            return RET_IGNORE;
        }

        __builtin_memcpy(&data->buf[data->state->tail_ctx.off & SCRATCH_SIZE_HALF], &saddr, 16);
        __builtin_memcpy(&data->buf[(data->state->tail_ctx.off + 16) & SCRATCH_SIZE_HALF], &sport, 2);
        __builtin_memcpy(&data->buf[(data->state->tail_ctx.off + 18) & SCRATCH_SIZE_HALF], daddr, 16);
        __builtin_memcpy(&data->buf[(data->state->tail_ctx.off + 34) & SCRATCH_SIZE_HALF], &dport, 2);

        data->state->tail_ctx.off += 16 + 2 + 16 + 2;
        break;
    }
    default:
        return RET_IGNORE;
    }

    return RET_SUCCESS;
}

static __always_inline int flush_filler(void* ctx, struct filler_data *data) {
    int len = data->state->tail_ctx.off;

    if (len <= 0) {
        // nothing to flush
        return RET_SUCCESS;
    }

    if (len > PERF_EVENT_MAX_SIZE) {
        return RET_FAILURE_BUFFER_FULL;
    }

    // TODO(buglloc): fix padding
    // make old verifier happy
    return bpf_perf_event_output(ctx, &events, BPF_F_CURRENT_CPU, data->buf, ((len - 1) & SCRATCH_SIZE_MAX) + 1);
}

#endif
