#ifndef __INTERNAL_H
#define __INTERNAL_H

#include "bpf_helpers.h"

#include <linux/ptrace.h> // Y_IGNORE
#include <linux/sched.h> // Y_IGNORE
#include <linux/cgroup.h> // Y_IGNORE
#include <linux/ipv6.h> // Y_IGNORE
#include <linux/in.h> // Y_IGNORE
#include <linux/fdtable.h> // Y_IGNORE
#include <linux/net.h> // Y_IGNORE
#include <linux/signal.h> // Y_IGNORE
#include <linux/tty.h> // Y_IGNORE
#include <generated/uapi/linux/version.h> // Y_IGNORE
#include <net/compat.h> // Y_IGNORE
#include <net/sock.h> // Y_IGNORE
#include <net/inet_sock.h> // Y_IGNORE
#include <net/af_unix.h> // Y_IGNORE


struct perf_event_header {
    u32 type;
    u16 misc;
    u16 size;
};

struct perf_event_sample {
    struct perf_event_header header;
    u32 size;
    char data[];
};

/*
 * Unfortunately the entire perf event length must fit in u16
 */
#define PERF_EVENT_MAX_SIZE (0xffff - sizeof(struct perf_event_sample))

#define RET_SUCCESS 0
#define RET_FAILURE_BUFFER_FULL -1
#define RET_FAILURE_INVALID_USER_MEMORY -2
#define RET_FAILURE_BUG -3
#define RET_IGNORE -4

#define MAX_ARG_SIZE 4096
#define MAX_FAIL_ARG 10
#define SCRATCH_SIZE (1 << 14)
#define SCRATCH_SIZE_MAX (SCRATCH_SIZE - 1)
#define SCRATCH_SIZE_HALF (SCRATCH_SIZE_MAX >> 1)

struct tail_context {
    unsigned long off;
} __attribute__((packed));

struct bpf_per_cpu_state {
    struct tail_context tail_ctx;
} __attribute__((packed));

struct filler_data {
    struct bpf_per_cpu_state *state;
    char *buf;
};

struct bpf_map_def events _section("maps") = {
    .type = BPF_MAP_TYPE_PERF_EVENT_ARRAY,
    .key_size = sizeof(u32),
    .value_size = sizeof(u32),
    .max_entries = 0,
};

struct bpf_map_def local_state_map _section("maps") = {
    .type = BPF_MAP_TYPE_PERCPU_ARRAY,
    .key_size = sizeof(u32),
    .value_size = sizeof(struct bpf_per_cpu_state),
    .max_entries = 1,
};

struct bpf_map_def frame_scratch_map _section("maps") = {
    .type = BPF_MAP_TYPE_PERCPU_ARRAY,
    .key_size = sizeof(u32),
    .value_size = SCRATCH_SIZE,
    // 0xFFFFFFFE mean "possible CPUs"
    .max_entries = 1,
};

// kernfs dropped kernfs_node_id union: https://github.com/torvalds/linux/commit/67c0496e87d193b8356d2af49ab95e8a1b954b3c
#if LINUX_VERSION_CODE > KERNEL_VERSION(5, 6, 0)
static __always_inline u64 kernfs_node_id(struct kernfs_node *kn) {
    return BPF_LOAD(kn->id);
}
#else
static __always_inline u64 kernfs_node_id(struct kernfs_node *kn) {
    return BPF_LOAD(kn->id.id);
}
#endif

static __always_inline char* get_frame_scratch_area() {
    char* scratch_ptr;
    u32 key = 0;
    scratch_ptr = bpf_map_lookup_elem(&frame_scratch_map, &key);
    if (!scratch_ptr) {
        bpf_printk("frame scratch NULL\n");
    }

    return scratch_ptr;
}

static __always_inline struct bpf_per_cpu_state* get_local_state() {
    u32 key = 0;
    struct bpf_per_cpu_state* state;

    state = bpf_map_lookup_elem(&local_state_map, &key);
    if (!state) {
        bpf_printk("state NULL\n");
    }

    return state;
}

static __always_inline void reset_tail_ctx(struct bpf_per_cpu_state* state) {
    state->tail_ctx.off = 0;
}

static __always_inline u64 get_cgroup_id(struct css_set *cgroups) {
    struct cgroup_subsys_state *css = BPF_LOAD(cgroups->subsys[freezer_cgrp_id]);
    struct cgroup *cgroup = BPF_LOAD(css->cgroup);
    struct kernfs_node *kn = BPF_LOAD(cgroup->kn);
    return kernfs_node_id(kn);
}

static __always_inline struct file *get_task_fd(struct task_struct *task, int fd) {
	struct files_struct *files;
	struct fdtable *fdt;
	int max_fds;
	struct file **fds;

	files = BPF_LOAD(task->files);
	if (!files) {
    	return NULL;
	}

	fdt = BPF_LOAD(files->fdt);
	if (!fdt) {
	    return NULL;
	}

	max_fds = BPF_LOAD(fdt->max_fds);
	if (fd >= max_fds) {
	    return NULL;
	}

	fds = BPF_LOAD(fdt->fd);
	if (!fds) {
	    return NULL;
	}

	return BPF_LOAD(fds[fd]);
}

static __always_inline u8 read_sock_proto(struct sock *sk) {
    u8 protocol = 0;

    // holly shit, bpf_probe_read can't read bitfields :/
    // TODO(buglloc): do something better

    #if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
        bpf_probe_read(&protocol, 1, (void *)((u64)&sk->sk_gso_max_segs) - 3);
    #elif __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
        bpf_probe_read(&protocol, 1, (void *)((u64)&sk->sk_gso_max_segs) - 1);
    #else
    # error "Fix your compiler's __BYTE_ORDER__?!"
    #endif

    return protocol;
}

static __always_inline bool is_acceptable_sock(struct socket *sock, struct sockaddr_storage *skaddr) {
    u16 family;
    u8 proto;
    struct sock *sk;

    sk = BPF_LOAD(sock->sk);
    if (!sk) {
        return false;
    }

    proto = read_sock_proto(sk);
    if (proto != IPPROTO_TCP) {
        return false;
    }

    family = BPF_LOAD(sk->sk_family);
    switch (family) {
    case AF_INET:
    case AF_INET6:
        return true;
    default:
        return false;
    }
}

static __always_inline bool is_ipv6_addr_empty(const struct in6_addr *a) {
	const unsigned long *ul = (const unsigned long *)a;

	return (ul[0] | ul[1]) == 0UL;
}

static __always_inline struct socket *get_task_sockfd(struct task_struct *task, int fd) {
	struct file *file;

    if (fd < 0) {
        return NULL;
    }

	file = get_task_fd(task, fd);
	if (!file) {
	    return NULL;
	}

	return BPF_LOAD(file->private_data);
}

static __always_inline int bpf_addr_to_kernel(void *uaddr, int ulen, struct sockaddr_storage *kaddr) {
	if (ulen <= 0 || (ulen & 0xff) > sizeof(struct sockaddr_storage)) {
	    return -EINVAL;
	}

	if (bpf_probe_read(kaddr, ulen & 0xff, uaddr)) {
	    return -EFAULT;
	}

	return 0;
}

static __always_inline const char *get_task_tty_name(struct task_struct *task) {
	struct signal_struct *signal;
    struct tty_struct *tty;

    signal = BPF_LOAD(task->signal);
    if (!signal) {
        return NULL;
    }

    tty = BPF_LOAD(signal->tty);
    if (!tty) {
        return NULL;
    }

	return BPF_LOAD(tty->name);
}

#endif
