#ifndef __PROBES_SECCOMP_H
#define __PROBES_SECCOMP_H

#include "bpf_helpers.h"
#include "internal.h"
#include "filler.h"

#include <linux/sched.h> // Y_IGNORE
#include <linux/seccomp.h> // Y_IGNORE

/*
prctl_set_seccomp(
    unsigned long seccomp_mode,
    char __user *filter);
*/
_section("kprobe/prctl_set_seccomp")
int kprobe_prctl_set_seccomp(struct pt_regs *ctx) {
    struct task_struct *task;
    struct mm_struct *mm;
    struct seccomp task_seccomp;
    const char *tty_name;
    struct filler_data data;
    int ret;
    unsigned long arg_start, arg_end, env_start, env_end, title_len, env_len;
    unsigned long seccomp_mode = PT_REGS_ARG_0(ctx);
    if (seccomp_mode != SECCOMP_MODE_FILTER) {
        return 0;
    }

    task = (struct task_struct *)bpf_get_current_task();
    if (!task) {
        bpf_printk("kprobe_prctl_set_seccomp: no current task\n");
        goto cleanup;
    }

    task_seccomp = BPF_LOAD(task->seccomp);
    if (task_seccomp.mode != SECCOMP_MODE_DISABLED) {
        // sessionleader started w/o seccomp profile
        goto cleanup;
    }

    ret = init_filler_data(&data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_prctl_set_seccomp: init_filler_data fail: %d\n", ret);
        goto cleanup;
    }

    reset_tail_ctx(data.state);

    // we doesn't have session id for this kind of events
    ret = fill_event_header(&data, EVENT_KIND_NEW_SESSION);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_prctl_set_seccomp: fill_event_header fail: %d\n", ret);
        goto cleanup;
    }

    ret = fill_procinfo(&data, task);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_prctl_set_seccomp: fill_procinfo fail: %d\n", ret);
        goto cleanup;
    }

    mm = BPF_LOAD(task->mm);
    if (!mm) {
        bpf_printk("kprobe_prctl_set_seccomp: no task->mm\n");
        goto cleanup;
    }


    arg_start = BPF_LOAD(mm->arg_start);
    if (!arg_start) {
        bpf_printk("kprobe_prctl_set_seccomp: failed to read mm->arg_start\n");
        goto cleanup;
    }

    arg_end = BPF_LOAD(mm->arg_end);
    if (!arg_end) {
        bpf_printk("kprobe_prctl_set_seccomp: failed to read mm->arg_end\n");
        goto cleanup;
    }

    env_start = BPF_LOAD(mm->env_start);
    if (!env_start) {
        bpf_printk("kprobe_prctl_set_seccomp: failed to read mm->env_start\n");
        goto cleanup;
    }


    env_end = BPF_LOAD(mm->env_end);
    if (!env_end) {
        bpf_printk("kprobe_prctl_set_seccomp: failed to read mm->env_end\n");
        goto cleanup;
    }

// linux/fs/proc/base.c:
//   /*
//	 * We allow setproctitle() to overwrite the argument
//	 * strings, and overflow past the original end. But
//	 * only when it overflows into the environment area.
//	 */
    if (env_start != arg_end || env_end < env_start) {
        env_start = env_end = arg_end;
    }

    title_len = env_end - arg_start;
    if (title_len > 0) {
        ret = fill_data(&data, (unsigned long)arg_start, title_len, PT_CHARBUF);
        if (ret != RET_SUCCESS) {
            bpf_printk("kprobe_prctl_set_seccomp: failed to write proc args: %d\n", ret);
            goto cleanup;
        }
    } else {
         bpf_printk("kprobe_prctl_set_seccomp: invalid proctitle\n");
         goto cleanup;
    }

    env_len = env_end - env_start;
    if (env_len > 0) {
        // drop lastr null byte
        env_len--;
    }

    ret = fill_data(&data, (unsigned long)env_start, env_len, PT_BYTEBUF);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_prctl_set_seccomp: failed to write proc env: %d\n", ret);
        goto cleanup;
    }

    tty_name = get_task_tty_name(task);
    if (tty_name) {
        ret = fill_data_type(&data, (unsigned long)tty_name, PT_CHARBUF);
    } else {
        const char na[] = "n/a";
        ret = fill_data(&data, (unsigned long)na, sizeof(na), PT_CHARBUF);
    }

    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_prctl_set_seccomp: failed to tty name: %d\n", ret);
        goto cleanup;
    }

    fill_eof(&data);
    ret = flush_filler(ctx, &data);
    if (ret != RET_SUCCESS) {
        bpf_printk("kprobe_prctl_set_seccomp: flush_ring fail: %d\n", ret);
        goto cleanup;
    }
cleanup:
    return 0;
}

#endif
