#ifndef XNET_H
#define XNET_H

// borrowed from vasgen

/// Returns true if the given IPv4 address is a loopback address.
#define xnet_is_ipv4_loopback(addr) \
    (((u32)addr & 0x000000ff) == 0x7f)

/// Returns true if the given IPv6 address is a loopback address.
///
/// Okay, you might ask "what a wonderful piece of code is this?", here is the answer.
///
/// Suppose we have "::ffff:127.0.0.1" address. Internally it is represented
/// either as:
///   - [0x0, 0x100007fffff0000] - 64-bit words.
///   - [0x0, 0x0, 0xffff0000, 0x100007f] - 32-bit words.
///   - [0x0, 0x0, 0x0, 0x0, 0x0, 0xffff, 0x7f, 0x100] - 16-bit words.
///   - [0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0x7f, 0x0, 0x0, 0x1] - bytes.
///
/// Here is the Python script to play with:
///   [', '.join(hex(v) for v in struct.unpack('<QQ', socket.inet_pton(socket.AF_INET6, '::ffff:127.0.0.1')))]
///
/// To ensure that the specified address is a loopback we do the following:
///   - Compare addr[0:8] word with 0.
///   - Compare addr[8:16] word with 0x100000000000000 to catch the "::1" case.
///   - Otherwise compare addr[8:12] word with 0xffff0000
///     AND addr[12:16] & 0x000000ff with 0x7f to catch the "::ffff:127.x.x.x" case.
#define xnet_is_ipv6_loopback(addr) \
    ((*(u64 *)&addr[0] == 0) && ((*(u64 *)&addr[8] == 0x100000000000000) || \
                                ((*(u32 *)&addr[8] == 0xffff0000) && ((*(u32 *)&addr[12] & 0x000000ff) == 0x7f))))


#endif // XNET_H
