package e2e

import (
	"bytes"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"sync"
	"testing"

	"github.com/klauspost/compress/zstd"
	"github.com/stretchr/testify/assert"
	"go.uber.org/zap"
	"go.uber.org/zap/zaptest"

	"a.yandex-team.ru/library/go/core/log"
	xzap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
	"a.yandex-team.ru/security/gideon/internal/protoseq"
)

type HTTPCollector struct {
	*httptest.Server
	t      *testing.T
	l      log.Logger
	zstd   *zstd.Decoder
	events []*events.Event
	mu     sync.Mutex
}

func NewHTTPCollector(t *testing.T) *HTTPCollector {
	decoder, err := zstd.NewReader(nil)
	if err != nil {
		panic(err)
	}

	c := &HTTPCollector{
		t:    t,
		zstd: decoder,
		l: &xzap.Logger{
			L: zaptest.NewLogger(t).WithOptions(zap.Development()),
		},
	}
	c.Server = httptest.NewServer(c)
	return c
}

func (c *HTTPCollector) Events() []*events.Event {
	c.mu.Lock()
	defer c.mu.Unlock()

	return c.events
}

func (c *HTTPCollector) ServeHTTP(_ http.ResponseWriter, r *http.Request) {
	body := c.Decompress(r)
	dec := protoseq.NewDecoder(bytes.NewReader(body))
	for dec.More() {
		var e events.Event
		err := dec.Decode(&e)
		assert.NoError(c.t, err)

		c.l.Info("new event received", log.Any("event", e))

		c.mu.Lock()
		c.events = append(c.events, &e)
		c.mu.Unlock()
	}
}

func (c *HTTPCollector) Decompress(r *http.Request) []byte {
	data, err := ioutil.ReadAll(r.Body)
	assert.NoError(c.t, err)

	if r.Header.Get("Content-Encoding") != "zstd" {
		return data
	}

	out, err := c.zstd.DecodeAll(data, nil)
	assert.NoError(c.t, err)
	return out
}
