package e2e

import (
	"encoding/json"
	"fmt"
	"os"
	"reflect"
	"runtime"
	"strings"
	"syscall"
	"testing"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/cilium/ebpf/rlimit"
	"github.com/davecgh/go-spew/spew"
	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/stretchr/testify/require"
	"golang.org/x/sys/unix"

	"a.yandex-team.ru/library/go/test/yatest"
	"a.yandex-team.ru/security/gideon/fake-iss/isstest"
	"a.yandex-team.ru/security/gideon/gideon/e2e/son"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector/httpcollector"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector/stdcollector"
	"a.yandex-team.ru/security/gideon/gideon/internal/config"
	"a.yandex-team.ru/security/gideon/gideon/internal/kernel"
	"a.yandex-team.ru/security/gideon/gideon/internal/probe"
	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
	"a.yandex-team.ru/security/libs/go/porto"
)

var AllTests = []func(t *testing.T){
	TestGideonSystemWideStart,
	TestGideonSeccompStart,
	TestGideonMixedStart,
	TestGideonSystemWideExec,
	TestGideonSlotExec,
	TestGideonPortoshellExec,
	TestGideonPortoshellOpenWrite,
	TestGideonYtJobShellExec,
}

func RunAll(t *testing.T) {
	kernVer, err := kernel.CurrentVersion()
	require.NoError(t, err)
	t.Logf("e2e runned on: %s", kernVer)

	err = kernVer.CheckPrerequisites()
	require.NoError(t, err)

	err = rlimit.RemoveMemlock()
	require.NoError(t, err)

	for _, tc := range AllTests {
		t.Run(funcName(tc), tc)
	}
}

func TestGideonSystemWideStart(t *testing.T) {
	g := NewGideonApp(t,
		gideonConf(
			nil,
			[]probe.Probe{
				{
					Kind:        probe.KindProcExec,
					CollectKind: probe.CollectKindAny,
				},
				{
					Kind:        probe.KindSysConnect,
					CollectKind: probe.CollectKindAny,
				},
				{
					Kind:        probe.KindSysExec,
					CollectKind: probe.CollectKindAny,
				},
				{
					Kind:        probe.KindSysPtrace,
					CollectKind: probe.CollectKindAny,
				},
				{
					Kind:        probe.KindSysOpen,
					CollectKind: probe.CollectKindAny,
				},
			},
		),
	)
	defer g.Close()
}

func TestGideonSystemWideExec(t *testing.T) {
	sonPath, err := yatest.BinaryPath("security/gideon/gideon/e2e/son/cmd/son/son")
	require.NoError(t, err)

	cases := []*struct {
		container   portoContainer
		parseEvents func(t *testing.T, data []byte) []*events.Event
		events      []*events.Event
	}{
		{
			container: portoContainer{
				name: "exec-root",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind exec", sonPath),
					"user":    "root",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        0,
							Container:  "/porto/exec-root",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
		{
			container: portoContainer{
				name: "exec-nobody",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind exec", sonPath),
					"user":    "nobody",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        65534,
							Container:  "/porto/exec-nobody",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
		{
			container: portoContainer{
				name: "sub-exec-nobody",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind sub-exec", sonPath),
					"user":    "nobody",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.SubExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        65534,
							Container:  "/porto/sub-exec-nobody",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "sub-exec"},
							},
						},
					},
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.Child.PID,
							Name:       "son",
							Ppid:       sonInfo.Child.ParentPID,
							ParentName: "son",
							SessionId:  sonInfo.Child.SessionID,
							Uid:        65534,
							Container:  "/porto/sub-exec-nobody",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
		{
			// YP pod
			container: portoContainer{
				name: "ISS-AGENT--fn3utw2xamszrlbv/pod_agent_box_App",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind exec", sonPath),
					"user":    "root",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        0,
							Container:  "/porto/ISS-AGENT--fn3utw2xamszrlbv/pod_agent_box_App",
							PodId:      "fn3utw2xamszrlbv",
							PodSetId:   "cleanweb-synchronous-prod.rpc",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
		{
			// gencfg pod
			container: portoContainer{
				name: "ISS-AGENT--25014/25014_production_market_ir_ui_sas_hkuktvViN7N/iss_hook_start",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind exec", sonPath),
					"user":    "root",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        0,
							Container:  "/porto/ISS-AGENT--25014/25014_production_market_ir_ui_sas_hkuktvViN7N/iss_hook_start",
							PodId:      "25014",
							NannyId:    "production_market_ir_ui_sas",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
	}

	issSrv := isstest.NewServer(nil)
	defer issSrv.Close()

	collector := NewHTTPCollector(t)
	defer collector.Close()

	cfg := gideonConf(
		collector,
		[]probe.Probe{
			{
				Kind:        probe.KindProcExec,
				CollectKind: probe.CollectKindAny,
			},
		},
	)
	cfg.PodResolver.Enabled = true
	cfg.PodResolver.Upstream = issSrv.URL

	g := NewGideonApp(t, cfg)

	t.Run("start", func(t *testing.T) {
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				stdout := portoExec(t, tc.container)
				tc.events = tc.parseEvents(t, stdout)
			})
		}
	})

	// gideon MUST submits all inflight events on close
	g.Close()

	t.Run("check", func(t *testing.T) {
		allEvents := collector.Events()
		cmdOpts := cmp.Options{
			cmpopts.IgnoreFields(events.Event{}, "Ts"),
			cmpIgnoreAnyString(),
		}

		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				for _, expected := range tc.events {
					found := false
					for _, actual := range allEvents {
						if cmp.Equal(expected, actual, cmdOpts...) {
							found = true
							break
						}
					}

					require.Truef(t, found,
						"event not found. expected: %s\nall events:\n%s\n",
						spew.Sdump(expected), spew.Sdump(allEvents),
					)
				}
			})
		}
	})
}

func TestGideonSlotExec(t *testing.T) {
	sonPath, err := yatest.BinaryPath("security/gideon/gideon/e2e/son/cmd/son/son")
	require.NoError(t, err)

	cases := []*struct {
		container   portoContainer
		parseEvents func(t *testing.T, data []byte) []*events.Event
		events      []*events.Event
	}{
		{
			// YP pod
			container: portoContainer{
				name: "ISS-AGENT--fn3utw2xamszrlbv/pod_agent_box_App",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind exec", sonPath),
					"user":    "root",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        0,
							Container:  "/porto/ISS-AGENT--fn3utw2xamszrlbv/pod_agent_box_App",
							PodId:      "fn3utw2xamszrlbv",
							PodSetId:   "cleanweb-synchronous-prod.rpc",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
		{
			// gencfg pod
			container: portoContainer{
				name: "ISS-AGENT--25014/25014_production_market_ir_ui_sas_hkuktvViN7N/iss_hook_start",
				props: map[string]string{
					"command": fmt.Sprintf("%s --kind exec", sonPath),
					"user":    "root",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							Pid:        sonInfo.PID,
							Name:       "son",
							Ppid:       sonInfo.ParentPID,
							ParentName: "any",
							SessionId:  sonInfo.SessionID,
							Uid:        0,
							Container:  "/porto/ISS-AGENT--25014/25014_production_market_ir_ui_sas_hkuktvViN7N/iss_hook_start",
							PodId:      "25014",
							NannyId:    "production_market_ir_ui_sas",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
	}

	issSrv := isstest.NewServer(nil)
	defer issSrv.Close()

	collector := NewHTTPCollector(t)
	defer collector.Close()

	cfg := gideonConf(
		collector,
		[]probe.Probe{
			{
				Kind:        probe.KindProcExec,
				CollectKind: probe.CollectKindSlot,
			},
		},
	)
	cfg.PodResolver.Enabled = true
	cfg.PodResolver.Upstream = issSrv.URL

	g := NewGideonApp(t, cfg)

	t.Run("start", func(t *testing.T) {
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				stdout := portoExec(t, tc.container)
				tc.events = tc.parseEvents(t, stdout)
			})
		}
	})

	// gideon MUST submits all inflight events on close
	g.Close()

	t.Run("check", func(t *testing.T) {
		allEvents := collector.Events()
		cmdOpts := cmp.Options{
			cmpopts.IgnoreFields(events.Event{}, "Ts"),
			cmpIgnoreAnyString(),
		}

		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				for _, expected := range tc.events {
					found := false
					for _, actual := range allEvents {
						if cmp.Equal(expected, actual, cmdOpts...) {
							found = true
							break
						}
					}

					require.Truef(t, found,
						"event not found. expected: %s\nall events:\n%s\n",
						spew.Sdump(expected), spew.Sdump(allEvents),
					)
				}
			})
		}
	})
}

func TestGideonPortoshellExec(t *testing.T) {
	sonPath, err := yatest.BinaryPath("security/gideon/gideon/e2e/son/cmd/son/son")
	require.NoError(t, err)

	seccompPath, err := yatest.BinaryPath("infra/portoshell/sessionleader/sessionleader")
	require.NoError(t, err)

	cases := []*struct {
		container   portoContainer
		parseEvents func(t *testing.T, data []byte) []*events.Event
		events      []*events.Event
	}{
		{
			container: portoContainer{
				name: "exec-root",
				props: map[string]string{
					"command": fmt.Sprintf("%s tst-usr %s --kind exec", seccompPath, sonPath),
					"user":    "root",
					"env":     "PORTOSHELL_SESSION_ID=portoshell-session",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_SSH_SESSION,
						Proc: &events.ProcInfo{
							SessionId:  sonInfo.SessionID,
							Name:       "sessionleader",
							ParentName: "any",
							Uid:        0,
							Container:  "/porto/exec-root",
						},
						Details: &events.Event_SshSession{
							SshSession: &events.SSHSessionEvent{
								User: "tst-usr",
								Id:   "portoshell-session",
								Kind: events.SessionKind_SK_PORTOSHELL,
							},
						},
					},
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							SessionId:  sonInfo.SessionID,
							Name:       "son",
							ParentName: "sessionleader",
							Uid:        0,
							Container:  "/porto/exec-root",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
	}

	collector := NewHTTPCollector(t)
	defer collector.Close()

	g := NewGideonApp(t,
		gideonConf(
			collector,
			[]probe.Probe{
				{
					Kind:        probe.KindProcExec,
					CollectKind: probe.CollectKindSSH,
				},
			},
		),
	)

	t.Run("start", func(t *testing.T) {
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				stdout := portoExec(t, tc.container)
				tc.events = tc.parseEvents(t, stdout)
			})
		}
	})

	// gideon MUST submits all inflight events on close
	g.Close()

	t.Run("check", func(t *testing.T) {
		allEvents := collector.Events()
		cmpOpts := cmp.Options{
			cmpopts.IgnoreFields(events.Event{}, "Ts"),
			cmpopts.IgnoreFields(events.ProcInfo{}, "Pid", "Ppid"),
			cmpopts.IgnoreFields(events.SSHSessionEvent{}, "Tty"),
			cmpIgnoreAnyString(),
		}

		foundedEvents := 0
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				for _, expected := range tc.events {
					found := false
					for _, actual := range allEvents {
						if cmp.Equal(expected, actual, cmpOpts...) {
							found = true
							break
						}
					}

					require.Truef(t, found,
						"event not found. expected: %s\nall events:\n%s\n",
						spew.Sdump(expected), spew.Sdump(allEvents),
					)
					foundedEvents++
				}
			})
		}

		require.Len(t, allEvents, foundedEvents)
	})
}

func TestGideonYtJobShellExec(t *testing.T) {
	sonPath, err := yatest.BinaryPath("security/gideon/gideon/e2e/son/cmd/son/son")
	require.NoError(t, err)

	seccompPath, err := yatest.BinaryPath("security/gideon/seccompish/seccompish")
	require.NoError(t, err)

	cases := []*struct {
		container   portoContainer
		parseEvents func(t *testing.T, data []byte) []*events.Event
		events      []*events.Event
	}{
		{
			container: portoContainer{
				name: "exec-root",
				props: map[string]string{
					"command": fmt.Sprintf("%s -- %s --kind exec", seccompPath, sonPath),
					"user":    "root",
					"env":     "YT_SHELL_ID=jobshell-session",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.ExecResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_SSH_SESSION,
						Proc: &events.ProcInfo{
							SessionId:  sonInfo.SessionID,
							Name:       "seccompish",
							ParentName: "any",
							Uid:        0,
							Container:  "/porto/exec-root",
						},
						Details: &events.Event_SshSession{
							SshSession: &events.SSHSessionEvent{
								Id:   "jobshell-session",
								Kind: events.SessionKind_SK_YT_JOBSHELL,
							},
						},
					},
					{
						Kind: events.EventKind_EK_PROC_EXEC,
						Proc: &events.ProcInfo{
							SessionId:  sonInfo.SessionID,
							Name:       "son",
							ParentName: "portoinit",
							Uid:        0,
							Container:  "/porto/exec-root",
						},
						Details: &events.Event_ProcExec{
							ProcExec: &events.ProcExecEvent{
								Exe:  sonPath,
								Args: []string{sonPath, "--kind", "exec"},
							},
						},
					},
				}
			},
		},
	}

	collector := NewHTTPCollector(t)
	defer collector.Close()

	g := NewGideonApp(t,
		gideonConf(
			collector,
			[]probe.Probe{
				{
					Kind:        probe.KindProcExec,
					CollectKind: probe.CollectKindSSH,
				},
			},
		),
	)

	t.Run("start", func(t *testing.T) {
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				stdout := portoExec(t, tc.container)
				tc.events = tc.parseEvents(t, stdout)
			})
		}
	})

	// gideon MUST submits all inflight events on close
	g.Close()

	t.Run("check", func(t *testing.T) {
		allEvents := collector.Events()
		cmpOpts := cmp.Options{
			cmpopts.IgnoreFields(events.Event{}, "Ts"),
			cmpopts.IgnoreFields(events.ProcInfo{}, "Pid", "Ppid"),
			cmpopts.IgnoreFields(events.SSHSessionEvent{}, "Tty"),
			cmpIgnoreAnyString(),
		}

		foundedEvents := 0
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				for _, expected := range tc.events {
					found := false
					for _, actual := range allEvents {
						if cmp.Equal(expected, actual, cmpOpts...) {
							found = true
							break
						}
					}

					require.Truef(t, found,
						"event not found. expected: %s\nall events:\n%s\n",
						spew.Sdump(expected), spew.Sdump(allEvents),
					)
					foundedEvents++
				}
			})
		}

		require.Len(t, allEvents, foundedEvents)
	})
}

func TestGideonPortoshellOpenWrite(t *testing.T) {
	sonPath, err := yatest.BinaryPath("security/gideon/gideon/e2e/son/cmd/son/son")
	require.NoError(t, err)

	seccompPath, err := yatest.BinaryPath("infra/portoshell/sessionleader/sessionleader")
	require.NoError(t, err)

	cases := []*struct {
		container   portoContainer
		parseEvents func(t *testing.T, data []byte) []*events.Event
		events      []*events.Event
	}{
		{
			container: portoContainer{
				name: "open-write-root",
				props: map[string]string{
					"command": fmt.Sprintf("%s tst-usr %s --kind open-write", seccompPath, sonPath),
					"user":    "root",
					"env":     "PORTOSHELL_SESSION_ID=portoshell-session",
				},
			},
			parseEvents: func(t *testing.T, data []byte) []*events.Event {
				var sonInfo son.OpenWriteResult
				err := json.Unmarshal(data, &sonInfo)
				require.NoError(t, err)

				return []*events.Event{
					{
						Kind: events.EventKind_EK_SSH_SESSION,
						Proc: &events.ProcInfo{
							SessionId:  sonInfo.SessionID,
							Name:       "sessionleader",
							ParentName: "any",
							Uid:        0,
							Container:  "/porto/open-write-root",
						},
						Details: &events.Event_SshSession{
							SshSession: &events.SSHSessionEvent{
								User: "tst-usr",
								Id:   "portoshell-session",
								Kind: events.SessionKind_SK_PORTOSHELL,
							},
						},
					},
					{
						Kind: events.EventKind_EK_SYS_OPEN_AT,
						Proc: &events.ProcInfo{
							SessionId:  sonInfo.SessionID,
							Name:       "son",
							ParentName: "sessionleader",
							Uid:        0,
							Container:  "/porto/open-write-root",
						},
						Details: &events.Event_OpenAt{
							OpenAt: &events.OpenAtEvent{
								RetCode:  0,
								Fd:       unix.AT_FDCWD,
								Filename: sonInfo.Filename,
								Flags:    int32(os.O_RDWR | os.O_CREATE | os.O_EXCL | syscall.O_CLOEXEC),
							},
						},
					},
				}
			},
		},
	}

	collector := NewHTTPCollector(t)
	defer collector.Close()

	g := NewGideonApp(t,
		gideonConf(
			collector,
			[]probe.Probe{
				{
					Kind:        probe.KindSysOpen,
					CollectKind: probe.CollectKindSSH,
				},
			},
		),
	)

	t.Run("start", func(t *testing.T) {
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				stdout := portoExec(t, tc.container)
				tc.events = tc.parseEvents(t, stdout)
			})
		}
	})

	// gideon MUST submits all inflight events on close
	g.Close()

	t.Run("check", func(t *testing.T) {
		allEvents := collector.Events()
		cmpOpts := cmp.Options{
			cmpopts.IgnoreFields(events.Event{}, "Ts"),
			cmpopts.IgnoreFields(events.ProcInfo{}, "Pid", "Ppid"),
			cmpopts.IgnoreFields(events.SSHSessionEvent{}, "Tty"),
			cmpIgnoreAnyString(),
		}

		foundedEvents := 0
		for _, tc := range cases {
			t.Run(tc.container.name, func(t *testing.T) {
				for _, expected := range tc.events {
					found := false
					for _, actual := range allEvents {
						if cmp.Equal(expected, actual, cmpOpts...) {
							found = true
							break
						}
					}

					require.Truef(t, found,
						"event not found. expected: %s\nall events:\n%s\n",
						spew.Sdump(expected), spew.Sdump(allEvents),
					)
					foundedEvents++
				}
			})
		}

		require.Len(t, allEvents, foundedEvents)
	})
}

func TestGideonSeccompStart(t *testing.T) {
	g := NewGideonApp(t,
		gideonConf(
			nil,
			[]probe.Probe{
				{
					Kind:        probe.KindProcExec,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysConnect,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysExec,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysPtrace,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysOpen,
					CollectKind: probe.CollectKindSSH,
				},
			},
		),
	)
	defer g.Close()
}

func TestGideonMixedStart(t *testing.T) {
	g := NewGideonApp(t,
		gideonConf(
			nil,
			[]probe.Probe{
				{
					Kind:        probe.KindProcExec,
					CollectKind: probe.CollectKindAny,
				},
				{
					Kind:        probe.KindSysConnect,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysExec,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysPtrace,
					CollectKind: probe.CollectKindSSH,
				},
				{
					Kind:        probe.KindSysOpen,
					CollectKind: probe.CollectKindAny,
				},
			},
		),
	)
	defer g.Close()
}

func TestKernelVersion(t *testing.T, expected string) {
	v, err := kernel.CurrentVersion()
	require.NoError(t, err)
	require.Equal(t, expected, v.String())
}

func funcName(fn func(t *testing.T)) string {
	name := runtime.FuncForPC(reflect.ValueOf(fn).Pointer()).Name()
	if idx := strings.LastIndexByte(name, '.'); idx > 0 {
		return name[idx+1:]
	}

	return name
}

func gideonConf(collector *HTTPCollector, probes []probe.Probe) config.Config {
	cfg := config.Config{
		Debug:             true,
		BusyPrerequisites: false,
		API: config.API{
			Enabled: false,
		},
		Tracer: config.Tracer{
			Collector:        stdcollector.Name,
			PerCPUBufferSize: 524288,
			Probes:           probes,
		},
		Metrics: config.Metrics{
			Enabled: false,
		},
		PodResolver: config.PodResolver{
			Enabled: false,
		},
	}

	if collector != nil {
		cfg.Tracer.Collector = httpcollector.Name
		cfg.HTTPCollector = httpcollector.Config{
			Upstream:        collector.URL,
			FlushTimeout:    1 * time.Second,
			FlushInterval:   50 * time.Millisecond,
			MaxMemoryUsage:  1 << 26, // 64MB,
			WriteBufferSize: 1 << 20, // 1MB
			Compression:     true,
		}
	}
	return cfg
}

type portoContainer struct {
	name  string
	props map[string]string
}

func portoExec(t *testing.T, req portoContainer) []byte {
	var portoAPI *porto.API
	err := backoff.Retry(func() (err error) {
		portoAPI, err = porto.NewAPI(nil)
		return
	}, backoff.WithMaxRetries(backoff.NewConstantBackOff(1*time.Second), 10))
	require.NoError(t, err)

	defer func() {
		_ = portoAPI.Close()
	}()

	parts := strings.Split(req.name, "/")
	if len(parts) > 1 {
		var cur []string
		for _, part := range parts[:len(parts)-1] {
			cur = append(cur, part)
			c, err := portoAPI.CreateWeakContainer(strings.Join(cur, "/"))
			require.NoErrorf(t, err, "create container %q", strings.Join(cur, "/"))
			// don't create new pid ns
			_ = c.SetProperty("isolate", "false")
		}
	}

	c, err := portoAPI.CreateWeakContainer(req.name)
	require.NoError(t, err)

	defer func() {
		_ = c.Destroy()
	}()

	// don't create new pid ns
	err = c.SetProperty("isolate", "false")
	require.NoError(t, err)
	for k, v := range req.props {
		err = c.SetProperty(k, v)
		require.NoError(t, err)
	}

	err = c.Start()
	require.NoError(t, err)

	for {
		_, err = c.Wait(10 * time.Second)
		require.NoError(t, err)

		state, err := c.GetState()
		if err != nil || state == porto.StateDead || state == porto.StateStopped {
			break
		}
	}

	status, err := c.GetProperty("exit_status")
	require.NoError(t, err)

	stderr, _ := c.GetProperty("stderr")
	require.Equal(t, "0", status, "son failed: %s", stderr)

	stdout, err := c.GetProperty("stdout")
	require.NoError(t, err)

	return []byte(stdout)
}

func cmpIgnoreAnyString() cmp.Option {
	any := func(s string) bool {
		return s == "any"
	}

	notAny := func(s string) bool {
		return s != "any" && s != ""
	}

	return cmp.FilterValues(func(a, b string) bool {
		return any(a) && notAny(b) || any(b) && notAny(a)
	}, cmp.Ignore())
}
