package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"strconv"
	"strings"

	"github.com/spf13/pflag"

	"a.yandex-team.ru/security/gideon/gideon/e2e/son"
)

func getSessionID() uint32 {
	data, err := ioutil.ReadFile("/proc/self/sessionid")
	if err != nil {
		panic(err)
	}

	id, err := strconv.ParseUint(string(data), 10, 64)
	if err != nil {
		panic(err)
	}

	return uint32(id)
}

func doExecTest() son.ExecResult {
	return son.ExecResult{
		PID:       uint32(os.Getpid()),
		ParentPID: uint32(os.Getppid()),
		UID:       uint32(os.Getuid()),
		SessionID: getSessionID(),
	}
}

func doSubExec() son.SubExecResult {
	ex, err := os.Executable()
	if err != nil {
		panic(err)
	}

	cmd := exec.Command(ex, "--kind", "exec")
	var stdout bytes.Buffer
	var stderr strings.Builder
	cmd.Stdout = &stdout
	cmd.Stderr = &stderr

	if err := cmd.Run(); err != nil {
		panic(fmt.Sprintf("%s\nstdout:\n%s\nstderr:\n%s", err, stdout.String(), stderr.String()))
	}

	var childRes son.ExecResult
	if err := json.Unmarshal(stdout.Bytes(), &childRes); err != nil {
		panic(fmt.Sprintf("%s\nstdout:\n%s\nstderr:\n%s", err, stdout.String(), stderr.String()))
	}

	return son.SubExecResult{
		ExecResult: doExecTest(),
		Child:      childRes,
	}
}

func doOpenWrite() son.OpenWriteResult {
	file, err := ioutil.TempFile("", "son-do-open-write")
	if err != nil {
		panic(fmt.Sprintf("create temp file: %v", err))
	}

	defer func() { _ = os.Remove(file.Name()) }()

	_, _ = file.Write([]byte{'k', 'e', 'k'})
	return son.OpenWriteResult{
		ExecResult: son.ExecResult{
			PID:       uint32(os.Getpid()),
			ParentPID: uint32(os.Getppid()),
			UID:       uint32(os.Getuid()),
			SessionID: getSessionID(),
		},
		FD:       int64(file.Fd()),
		Filename: file.Name(),
	}
}

func main() {
	var kind string
	pflag.StringVar(&kind, "kind", "exec", "test kind")
	pflag.Parse()

	var result interface{}
	switch kind {
	case "exec":
		result = doExecTest()
	case "sub-exec":
		result = doSubExec()
	case "open-write":
		result = doOpenWrite()
	default:
		panic(fmt.Sprintf("unsupported test kind: %s", kind))
	}

	err := json.NewEncoder(os.Stdout).Encode(result)
	if err != nil {
		panic(err)
	}
}
