package app

import (
	"context"
	"fmt"
	"net"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector"
	"a.yandex-team.ru/security/gideon/gideon/internal/config"
	"a.yandex-team.ru/security/gideon/gideon/internal/podresolver"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
	"a.yandex-team.ru/security/gideon/gideon/internal/tracer"
	"a.yandex-team.ru/security/gideon/gideon/internal/yasm"
)

type App struct {
	cfg         *config.Config
	ctx         context.Context
	cancelCtx   context.CancelFunc
	tracer      *tracer.Tracer
	collector   collector.Collector
	unistat     *sensors.UnistatSensor
	yasmPusher  *yasm.Pusher
	http        *http.Server
	podResolver *podresolver.Resolver
	log         log.Logger
}

func NewApp(cfg *config.Config, l log.Logger) (app *App, err error) {
	app = &App{
		cfg: cfg,
		log: l,
	}

	app.ctx, app.cancelCtx = context.WithCancel(context.Background())

	podResolverOpts := []podresolver.Option{
		podresolver.WithUpstream(cfg.PodResolver.Upstream),
		podresolver.WithLogger(l),
	}

	collectorOpts := []collector.Option{
		collector.WithLogger(l),
	}

	tracerOpts := []tracer.Option{
		tracer.WithRingBufferSize(cfg.Tracer.PerCPUBufferSize),
		tracer.WithModulePath(cfg.Tracer.CustomModule),
		tracer.WithProbe(cfg.Tracer.Probes...),
		tracer.WithDebug(cfg.Debug),
		tracer.WithLogger(l),
	}

	if cfg.Metrics.Enabled {
		app.unistat = sensors.NewUnistatSensors()
		app.yasmPusher = yasm.NewPusher(
			app.unistat,
			yasm.WithUpstream(cfg.Metrics.PushUpstream),
			yasm.WithYasmTags(cfg.Metrics.Tags),
			yasm.WithLogger(l),
		)

		tracerOpts = append(tracerOpts, tracer.WithMetrics(app.unistat))
		collectorOpts = append(collectorOpts, collector.WithMetrics(app.unistat))
		podResolverOpts = append(podResolverOpts, podresolver.WithMetrics(app.unistat))
	}

	if cfg.PodResolver.Enabled {
		app.podResolver, err = podresolver.NewResolver(podResolverOpts...)
		if err != nil {
			return nil, fmt.Errorf("failed to create pod resolver: %w", err)
		}

		tracerOpts = append(tracerOpts, tracer.WithPodResolver(app.podResolver))
	}

	app.collector, err = collector.NewByConfig(cfg, collectorOpts...)
	if err != nil {
		return nil, fmt.Errorf("failed to create collector: %w", err)
	}

	tracerOpts = append(tracerOpts, tracer.WithCollector(app.collector))
	app.tracer, err = tracer.NewTracer(tracerOpts...)
	if err != nil {
		return nil, fmt.Errorf("failed to create tracer: %w", err)
	}

	if cfg.API.Enabled {
		app.http = &http.Server{
			Handler:     newRouter(app),
			ConnContext: app.SavePeerCreds,
		}
	}

	return app, nil
}

func (a *App) Start(onStarted func()) error {
	// TODO(buglloc):  errgroup

	if err := a.tracer.InitBPF(); err != nil {
		return fmt.Errorf("failed to initialize tracer BPF: %w", err)
	}

	if err := a.tracer.EnablePortoProbes(); err != nil {
		return fmt.Errorf("failed to enable cgroups probes: %w", err)
	}

	if err := a.tracer.InitConsumer(); err != nil {
		return fmt.Errorf("failed to initialize consumer: %w", err)
	}

	consumeChan := make(chan error, 1)
	go func() {
		if err := a.tracer.Consume(); err != nil {
			consumeChan <- fmt.Errorf("consume fail: %w", err)
		}
	}()

	if a.podResolver != nil {
		a.podResolver.Sync(a.ctx)
		go a.podResolver.CacheWatcher(a.ctx)
	}

	if err := a.tracer.Restore(); err != nil {
		return fmt.Errorf("failed to restore tracer state: %w", err)
	}

	if err := a.tracer.EnableProbes(); err != nil {
		return fmt.Errorf("failed to enable probes: %w", err)
	}

	if a.http != nil {
		a.log.Info("start http API", log.String("addr", a.http.Addr))
		go func() {
			listener, err := net.Listen("unix", a.cfg.API.Addr)
			if err != nil {
				a.log.Error("api listen fail", log.Error(err))
				return
			}

			defer func() {
				_ = listener.Close()
			}()

			if err := a.http.Serve(listener); err != nil {
				a.log.Error("api listen fail", log.Error(err))
			}
		}()
	}

	if a.yasmPusher != nil {
		go a.yasmPusher.Start(a.ctx, a.cfg.Metrics.PushPeriod)
	}

	a.log.Info("gideon started")
	if onStarted != nil {
		onStarted()
	}

	select {
	case <-a.ctx.Done():
		return nil
	case err := <-consumeChan:
		return err
	}
}

func (a *App) Shutdown(ctx context.Context) error {
	// first of all - stop tracer to stop event consuming
	a.tracer.Close(ctx)

	// then - collector
	a.collector.Close(ctx)

	// after all - stop api server
	if a.http != nil {
		_ = a.http.Shutdown(ctx)
	}

	// then - application context
	a.cancelCtx()
	return nil
}
