package app

import (
	"context"
	"encoding/json"
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/middleware/recovery"
	"a.yandex-team.ru/security/gideon/gideon/internal/config"
	"a.yandex-team.ru/security/libs/go/chim"
)

var (
	okRsp = []byte(`OK`)
)

type emitNewSessionReq struct {
	User string `json:"user"`
	TTY  string `json:"tty"`
}

type emitNewSessionRsp struct {
	ExternalSessionID string `json:"external_session_id"`
}

func newRouter(a *App) *chi.Mux {
	r := chi.NewRouter()
	r.Use(recovery.New(recovery.WithLogger(a.log.Structured())))
	r.Use(logRequest(a.log))

	r.Route("/admin/", func(r chi.Router) {
		r.Use(rootOnly)

		r.Mount("/debug", chim.Profiler("/admin"))
		r.Get("/stop", a.stopHandler)
	})

	r.Route("/tracer", func(r chi.Router) {
		r.Use(rootOnly)

		r.Post("/session", a.emitNewSession)
	})
	r.Get("/ping", a.pingHandler)
	return r
}

func (a *App) pingHandler(w http.ResponseWriter, _ *http.Request) {
	_, _ = w.Write(okRsp)
}

func (a *App) stopHandler(w http.ResponseWriter, _ *http.Request) {
	a.log.Info("received stop request")
	_, _ = w.Write(okRsp)

	go func() {
		ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
		defer cancel()

		err := a.Shutdown(ctx)
		if err != nil {
			a.log.Error("shutdown failed", log.Error(err))
		}
	}()
}

func (a *App) emitNewSession(w http.ResponseWriter, r *http.Request) {
	creds := credsFromContext(r.Context())
	if creds == nil {
		http.Error(w, "no creds from request", http.StatusBadRequest)
		return
	}

	var body emitNewSessionReq
	if err := json.NewDecoder(r.Body).Decode(&body); err != nil {
		http.Error(w, err.Error(), http.StatusBadRequest)
		return
	}

	sessID, err := a.tracer.EmitNewSession(uint32(creds.Pid), body.User, body.TTY)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	_ = json.NewEncoder(w).Encode(emitNewSessionRsp{
		ExternalSessionID: sessID,
	})
}
