package httpcollector

import (
	"context"
	"crypto/tls"
	"fmt"
	"io"
	"io/ioutil"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/gideon/gideon/internal/protoqueue"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
)

const Name = "HttpCollector"

type (
	HTTPCollector struct {
		queue   *protoqueue.ProtoQueue
		log     log.Logger
		metrics sensors.Sensor
		httpc   *resty.Client
	}

	Stats struct {
		// Number of records dropped during memory overflow.
		DroppedRecords int

		// Number of errors returned from underlying writer.
		WriteErrors int
	}
)

func NewCollector(cfg Config, opts ...Option) *HTTPCollector {
	c := &HTTPCollector{
		log:     &nop.Logger{},
		metrics: &sensors.NopSensor{},
	}

	for _, opt := range opts {
		opt(c)
	}

	c.queue = protoqueue.NewProtoQueue(
		c.queueSender,
		protoqueue.WithFlushInterval(cfg.FlushInterval),
		protoqueue.WithWriteBufferSize(cfg.WriteBufferSize),
		protoqueue.WithMaxSize(cfg.MaxMemoryUsage),
		protoqueue.WithLogger(c.log),
		protoqueue.WithMetrics(c.metrics),
		protoqueue.WithCompression(cfg.Compression),
	)

	c.httpc = resty.New().
		SetRetryCount(2).
		SetTimeout(cfg.FlushTimeout).
		SetLogger(c.log.Fmt()).
		SetHeader("Content-Type", "application/x-protobuf").
		SetDoNotParseResponse(true).
		SetHeader("User-Agent", "gideon <security@yandex-team.ru>").
		SetBaseURL(cfg.Upstream)

	if cfg.Compression {
		c.httpc.SetHeader("Content-Encoding", "zstd")
	}

	certPool, err := certifi.NewCertPool()
	if err != nil {
		c.log.Error("httpcollector: failed to create cert pool", log.Error(err))
	} else {
		c.httpc.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return c
}

func (c *HTTPCollector) NewEvent(event *events.Event) {
	if err := c.queue.EnqueueEvent(event); err != nil {
		c.log.Error("httpcollector: enqueue fail",
			log.Sprintf("type", "%T", event),
			log.Error(err))
	}
}

func (c *HTTPCollector) Close(ctx context.Context) {
	c.queue.Stop(ctx)
}

func (c *HTTPCollector) queueSender(ctx context.Context, data []byte) error {
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetBody(data).
		Post("")

	if err != nil {
		return fmt.Errorf("failed to send buffer: %w", err)
	}

	defer func() {
		_, _ = io.Copy(ioutil.Discard, rsp.RawResponse.Body)
		_ = rsp.RawBody().Close()
	}()

	if !rsp.IsSuccess() {
		return fmt.Errorf("non-success response: %d", rsp.StatusCode())
	}
	return nil
}
