package lbcollector

import (
	"context"
	"errors"
	"fmt"
	"os"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/gideon/gideon/internal/protoqueue"
	"a.yandex-team.ru/security/gideon/gideon/internal/secrets"
	"a.yandex-team.ru/security/gideon/gideon/internal/secrets/auto"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
	"a.yandex-team.ru/security/libs/go/lblight"
)

const Name = "LbCollector"

type LbCollector struct {
	log       log.Logger
	metrics   sensors.Sensor
	queue     *protoqueue.ProtoQueue
	writer    *lblight.Writer
	secrets   secrets.Config
	tvmClient tvm.Client
}

func NewCollector(cfg Config, opts ...Option) (*LbCollector, error) {
	if cfg.Topic == "" {
		return nil, errors.New("lbcollector: no topic configured")
	}

	c := &LbCollector{
		log:     &nop.Logger{},
		metrics: &sensors.NopSensor{},
	}

	for _, opt := range opts {
		opt(c)
	}

	sourceID, err := os.Hostname()
	if err != nil || sourceID == "" {
		sourceID = uuid.Must(uuid.NewV4()).String()
	}

	credentials, err := c.initAuth(cfg)
	if err != nil {
		return nil, fmt.Errorf("lbcollector: failed to initialize credentials: %w", err)
	}

	lbCodec := lblight.CodecRaw
	if cfg.Compression {
		lbCodec = lblight.CodecZstd
	}

	c.writer, err = lblight.NewWriter(
		cfg.Topic,
		lblight.WriterWithEndpoint(cfg.Endpoint),
		lblight.WriterWithLogger(c.log.WithName("persqueue")),
		lblight.WriterWithCreds(credentials),
		lblight.WriterWithSourceID(sourceID),
		lblight.WriterWithCodec(lbCodec),
	)
	if err != nil {
		return nil, fmt.Errorf("lbcollector: failed to create writer: %w", err)
	}

	err = c.writer.Dial(context.Background())
	if err != nil {
		_ = c.writer.Close(context.Background())
		return nil, fmt.Errorf("lbcollector: failed to initialize lb writer: %w", err)
	}

	go c.recvResponses()

	c.queue = protoqueue.NewProtoQueue(
		c.queueSender,
		protoqueue.WithFlushInterval(cfg.FlushInterval),
		protoqueue.WithWriteBufferSize(cfg.WriteBufferSize),
		protoqueue.WithMaxSize(cfg.MaxMemoryUsage),
		protoqueue.WithLogger(c.log),
		protoqueue.WithMetrics(c.metrics),
		protoqueue.WithCompression(cfg.Compression),
	)
	return c, nil
}

func (c *LbCollector) NewEvent(event *events.Event) {
	if err := c.queue.EnqueueEvent(event); err != nil {
		c.log.Error("lbcollector: enqueue fail",
			log.Sprintf("type", "%T", event),
			log.Error(err))
	}
}

func (c *LbCollector) Close(ctx context.Context) {
	c.queue.Stop(ctx)
	err := c.writer.Close(ctx)
	if err != nil {
		c.log.Error("lbcollector: failed to close lb writer", log.Error(err))
	}
}

func (c *LbCollector) recvResponses() {
	for {
		msg, err := c.writer.FetchFeedback()
		if err == lblight.ErrClosed {
			return
		}

		if err != nil {
			c.metrics.CollectorErrors(1)
			continue
		}

		switch m := msg.(type) {
		case *lblight.FeedbackMsgError:
			c.metrics.CollectorErrors(1)
			c.log.Error("lbcollector: issue feedback", log.Error(m.Error))
		default:
		}
	}
}

func (c *LbCollector) queueSender(ctx context.Context, data []byte) error {
	msg := lblight.WriteMsg{
		Data:      data,
		Timestamp: time.Now(),
	}

	return c.writer.Write(ctx, msg)
}

func (c *LbCollector) initAuth(cfg Config) (ydb.Credentials, error) {
	if cfg.AuthKind == AuthKindNone {
		return nil, nil
	}

	sp, err := auto.NewProvider(c.secrets)
	if err != nil {
		return nil, err
	}

	defer func() {
		_ = sp.Close()
	}()

	switch cfg.AuthKind {
	case AuthKindOAuth:
		return c.oAuthCredentials(cfg, sp)
	//case AuthKindTVM:
	//	return c.tvmCredentials(cfg, sp)
	default:
		return nil, fmt.Errorf("unsupported auth kind: %d", cfg.AuthKind)
	}
}

func (c *LbCollector) oAuthCredentials(cfg Config, sp secrets.Provider) (ydb.Credentials, error) {
	if cfg.OAuthToken == "" {
		return nil, errors.New("empty oauth_token")
	}

	oauthToken, err := sp.GetSecret(context.Background(), cfg.OAuthToken)
	if err != nil {
		return nil, fmt.Errorf("failed to get oauth token: %w", err)
	}

	return &ydb.AuthTokenCredentials{
		AuthToken: oauthToken,
	}, nil
}

//func (c *LbCollector) tvmCredentials(cfg Config, sp secrets.Provider) (ydb.Credentials, error) {
//	if cfg.TVM.SelfID == 0 {
//		return nil, errors.New("TVM self_id required")
//	}
//
//	if cfg.TVM.Secret == "" {
//		return nil, errors.New("TVM secret required")
//	}
//
//	tvmSecret, err := sp.GetSecret(context.Background(), cfg.TVM.Secret)
//	if err != nil {
//		return nil, fmt.Errorf("failed to get TVM secret: %w", err)
//	}
//
//	return newAuthProvider(cfg.TVM.SelfID, tvmSecret, c.log)
//}
