package lbcollector

import (
	"context"
	"fmt"
	"os"
	"path/filepath"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/libs/go/ydbtvm"
)

const tvmTTL = 60 * time.Minute

type authProvider struct {
	tvmClient *tvmauth.Client
	ticket    string
	expire    time.Time
}

func newAuthProvider(selfID tvm.ClientID, secret string, l log.Logger) (*authProvider, error) {
	cacheDir := filepath.Join(os.TempDir(), "gideon", "tvm")
	err := os.MkdirAll(cacheDir, 0700)
	if err != nil {
		l.Warn("can't create tvm cache dir", log.String("path", cacheDir), log.Error(err))
		cacheDir = ""
	}

	tvmSettings := tvmauth.TvmAPISettings{
		SelfID:                      selfID,
		EnableServiceTicketChecking: false,
		ServiceTicketOptions: tvmauth.NewIDsOptions(
			secret,
			[]tvm.ClientID{
				ydbtvm.LbClientID,
			},
		),
		DiskCacheDir: cacheDir,
	}

	tvmClient, err := tvmauth.NewAPIClient(tvmSettings, l.WithName("tvm"))
	if err != nil {
		return nil, fmt.Errorf("failed to create TVM client: %w", err)
	}

	return &authProvider{
		tvmClient: tvmClient,
	}, nil
}

func (a *authProvider) Token(ctx context.Context) (string, error) {
	if a.ticket != "" && a.expire.After(time.Now()) {
		return a.ticket, nil
	}

	ticket, err := a.tvmClient.GetServiceTicketForID(ctx, ydbtvm.LbClientID)
	if err != nil {
		return "", err
	}

	a.ticket = ticket
	a.expire = time.Now().Add(tvmTTL)
	return ticket, nil
}
