package commands

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/cilium/ebpf/rlimit"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/gideon/internal/app"
	"a.yandex-team.ru/security/gideon/gideon/internal/config"
	"a.yandex-team.ru/security/gideon/gideon/internal/kernel"
)

var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "Start Gideon",
	RunE: func(cmd *cobra.Command, args []string) error {
		kern, err := kernel.CurrentVersion()
		if err != nil {
			return fmt.Errorf("failed to get kernel version: %w", err)
		}

		if err := kern.CheckPrerequisites(); err != nil {
			if !cfg.BusyPrerequisites {
				return err
			}

			logger.Error("prerequisites doesn't satisfied, enter busy loop", log.Error(err))
			stopChan := make(chan os.Signal, 1)
			signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)
			<-stopChan
			return nil
		}

		if err := rlimit.RemoveMemlock(); err != nil {
			logger.Warn("failed to adjust rlimit memlock, gideon may fail", log.Error(err))
		}

		instance, err := app.NewApp(cfg, logger)
		if err != nil {
			return fmt.Errorf("failed to create application: %w", err)
		}

		errChan := make(chan error, 1)
		okChan := make(chan struct{}, 1)
		go func() {
			if err := instance.Start(nil); err != nil {
				errChan <- fmt.Errorf("failed to start application: %w", err)
			} else {
				okChan <- struct{}{}
			}
		}()

		stopChan := make(chan os.Signal, 1)
		signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

		defer logger.Info("Gideon stopped")

		select {
		case <-stopChan:
			logger.Info("shutting down gracefully by signal")

			ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
			defer cancel()

			return instance.Shutdown(ctx)
		case <-okChan:
			return nil
		case err := <-errChan:
			return err
		}
	},
}

func init() {
	rootCmd.AddCommand(startCmd)
}
