package commands

import (
	"errors"
	"fmt"
	"net"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/spf13/cobra"
)

var stopFlags = struct {
	Wait bool
}{
	Wait: false,
}

var stopCmd = &cobra.Command{
	Use:          "stop",
	SilenceUsage: true,
	Short:        "Gracefully stops Gideon",
	RunE: func(cmd *cobra.Command, args []string) error {
		if !cfg.API.Enabled {
			return errors.New("can't ping Gideon w/o API enabled")
		}

		_, p, err := net.SplitHostPort(cfg.API.Addr)
		if err != nil {
			return fmt.Errorf("failed to parse api addr: %w", err)
		}

		httpc := resty.New().
			SetTimeout(500 * time.Millisecond).
			SetBaseURL(fmt.Sprintf("http://localhost:%s", p))

		stop := func() error {
			logger.Info("sending stop request")
			rsp, err := httpc.R().Get("/admin/stop")
			if err != nil {
				return fmt.Errorf("gideon stop failed: %w", err)
			}

			if !rsp.IsSuccess() {
				return fmt.Errorf("gideon stop failed: unexpected status code: %d", rsp.StatusCode())
			}
			return nil
		}

		ping := func() error {
			_, err := httpc.R().Get("/ping")
			if err != nil {
				return fmt.Errorf("gideon ping failed: %w", err)
			}

			return nil
		}

		if err := stop(); err != nil {
			return err
		}

		// and wait it if needed
		if stopFlags.Wait {
			for {
				logger.Info("wait termination")
				if err := ping(); err != nil {
					// done
					fmt.Println("Gideon stopped: ping not respond")
					break
				}
				time.Sleep(500 * time.Millisecond)
			}
		} else {
			fmt.Println("Stop request was sent")
		}

		return nil
	},
}

func init() {
	stopCmd.PersistentFlags().BoolVar(&stopFlags.Wait, "wait", false, "wait termination")
	rootCmd.AddCommand(stopCmd)
}
