package config

import (
	"context"
	"fmt"
	"os"
	"strings"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector/httpcollector"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector/lbcollector"
	"a.yandex-team.ru/security/gideon/gideon/internal/probe"
	"a.yandex-team.ru/security/gideon/gideon/internal/secrets"
	"a.yandex-team.ru/security/gideon/gideon/internal/yasm"
)

const (
	ShutdownDeadline = 10 * time.Second
)

type Logger struct {
	Output string `yaml:"output"`
	Level  string `yaml:"level"`
}

type Tracer struct {
	CustomModule     string        `yaml:"custom_module"`
	PerCPUBufferSize int           `yaml:"per_cpu_buffer_size"`
	Collector        string        `yaml:"collector"`
	Probes           []probe.Probe `yaml:"probes"`
}

type API struct {
	Enabled bool   `yaml:"enabled"`
	Addr    string `yaml:"addr"`
}

type Config struct {
	Debug             bool                 `yaml:"debug"`
	BusyPrerequisites bool                 `yaml:"busy_prerequisites"`
	MLockLimit        uint64               `yaml:"mem_lock_limit"`
	Secrets           secrets.Config       `yaml:"secrets"`
	Logger            Logger               `yaml:"logger"`
	API               API                  `yaml:"api"`
	Tracer            Tracer               `yaml:"tracer"`
	Metrics           Metrics              `yaml:"metrics"`
	HTTPCollector     httpcollector.Config `yaml:"http_collector"`
	LBCollector       lbcollector.Config   `yaml:"lb_collector"`
	PodResolver       PodResolver          `yaml:"pod_resolver"`
}

func NewConfig(configPath string) (*Config, error) {
	cfg := &Config{
		Debug:             false,
		BusyPrerequisites: false,
		MLockLimit:        128 << 20,
		Logger: Logger{
			Output: "stderr",
			Level:  log.InfoString,
		},
		Secrets: secrets.Config{
			Kind: secrets.ProviderKindEnv,
			Secrets: map[string]string{
				"LB_TOKEN": "LB_TOKEN",
			},
			CrtPath:  "/etc/certs/capi.pem",
			FilePath: "/etc/gideon/secrets",
		},
		API: API{
			Enabled: false,
			Addr:    "@gideon",
		},
		Tracer: Tracer{
			PerCPUBufferSize: 524288,
			Collector:        "StdCollector",
			Probes: []probe.Probe{
				{
					Kind:        probe.KindProcExec,
					CollectKind: probe.CollectKindAny,
				},
			},
		},
		Metrics: Metrics{
			Enabled:      false,
			PushUpstream: yasm.DefaultUpstream,
			PushPeriod:   5 * time.Second,
		},
		HTTPCollector: httpcollector.Config{
			Upstream:        "https://blackhole.sec.yandex.net/logger",
			FlushTimeout:    5 * time.Second,
			FlushInterval:   500 * time.Millisecond,
			MaxMemoryUsage:  1 << 26, // 64MB,
			WriteBufferSize: 1 << 20, // 1MB
			Compression:     true,
		},
		LBCollector: lbcollector.Config{
			Endpoint:        "logbroker.yandex.net",
			FlushInterval:   500 * time.Millisecond,
			MaxMemoryUsage:  1 << 26, // 64MB
			WriteBufferSize: 1 << 20, // 1MB
			OAuthToken:      "LB_TOKEN",
			Compression:     true,
		},
		PodResolver: PodResolver{
			Enabled:  false,
			Upstream: "http://localhost:25536",
		},
	}

	if configPath == "" {
		return cfg, nil
	}

	loader := confita.NewLoader(file.NewBackend(configPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	cfg.Tracer.CustomModule = os.ExpandEnv(cfg.Tracer.CustomModule)
	return cfg, nil
}

func (l *Logger) ZapConfig() zap.Config {
	var lvl zapcore.Level
	switch strings.ToLower(l.Level) {
	case log.TraceString:
		lvl = zapcore.DebugLevel
	case log.DebugString:
		lvl = zapcore.DebugLevel
	case log.InfoString:
		lvl = zapcore.InfoLevel
	case log.WarnString:
		lvl = zapcore.WarnLevel
	case log.ErrorString:
		lvl = zapcore.ErrorLevel
	case log.FatalString:
		lvl = zapcore.FatalLevel
	default:
		lvl = zapcore.InfoLevel
	}

	return zap.Config{
		Level:            zap.NewAtomicLevelAt(lvl),
		Encoding:         "kv",
		OutputPaths:      []string{l.Output},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			NameKey:        "logger",
			LineEnding:     zapcore.DefaultLineEnding,
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.SecondsDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}
}
