package ebpfutil

import (
	"bytes"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"strconv"
)

const (
	kprobeEventsFileName = "/sys/kernel/debug/tracing/kprobe_events"
	kprobeEventTmpl      = "/sys/kernel/debug/tracing/events/kprobes/%s/id"
)

var ErrKprobeIDNotExist = errors.New("kprobe id file doesn't exist")

func WriteKprobeEvent(probeType, eventName, funcName, maxactiveStr string) (int, error) {
	f, err := os.OpenFile(kprobeEventsFileName, os.O_APPEND|os.O_WRONLY, 0666)
	if err != nil {
		return -1, fmt.Errorf("cannot open kprobe_events: %w", err)
	}
	defer func() { _ = f.Close() }()

	cmd := fmt.Sprintf("%s%s:%s %s\n", probeType, maxactiveStr, eventName, funcName)
	if _, err = f.WriteString(cmd); err != nil {
		return -1, fmt.Errorf("cannot write %q to kprobe_events: %w", cmd, err)
	}

	kprobeIDFile := fmt.Sprintf(kprobeEventTmpl, eventName)
	kprobeIDBytes, err := ioutil.ReadFile(kprobeIDFile)
	if err != nil {
		if os.IsNotExist(err) {
			return -1, ErrKprobeIDNotExist
		}
		return -1, fmt.Errorf("cannot read kprobe id: %w", err)
	}

	kprobeID, err := strconv.Atoi(string(bytes.TrimSpace(kprobeIDBytes)))
	if err != nil {
		return -1, fmt.Errorf("invalid kprobe id: %w", err)
	}

	return kprobeID, nil
}

func RemoveKprobeEvent(eventName string) error {
	f, err := os.OpenFile(kprobeEventsFileName, os.O_APPEND|os.O_WRONLY, 0666)
	if err != nil {
		return fmt.Errorf("cannot open kprobe_events: %w", err)
	}
	defer func() { _ = f.Close() }()

	cmd := fmt.Sprintf("-:%s\n", eventName)
	if _, err = f.WriteString(cmd); err != nil {
		return fmt.Errorf("cannot write %q to kprobe_events: %w", cmd, err)
	}
	return nil
}
