package ebpfutil

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"strconv"
	"sync"
	"syscall"
	"unsafe"

	"golang.org/x/sys/unix"
)

const (
	pmuTypeFile        = "/sys/bus/event_source/devices/%s/type"
	perfTypeTracepoint = "tracepoint"
	perfTypeKProbe     = "kprobe"
)

var perfTypes sync.Map

func PerfEventOpenTracepoint(id int, progFd int) (int, error) {
	typ, err := findPerfEventType(perfTypeTracepoint)
	if err != nil {
		return 0, err
	}

	attr := unix.PerfEventAttr{
		Type:        typ,
		Config:      uint64(id),
		Sample_type: unix.PERF_SAMPLE_RAW,
		Sample:      1,
		Wakeup:      1,
	}

	pfd, err := unix.PerfEventOpen(&attr, -1, 0, -1, unix.PERF_FLAG_FD_CLOEXEC)
	if err != nil {
		return -1, fmt.Errorf("unable to open perf events: %w", err)
	}

	if err := attachTracingEvent(pfd, progFd); err != nil {
		return -1, err
	}

	return pfd, nil
}

// https://github.com/torvalds/linux/commit/e12f03d7031a977356e3d7b75a68c2185ff8d155
// https://github.com/iovisor/bcc/blob/6e9b4509fc7a063302b574520bac6d49b01ca97e/src/cc/libbpf.c#L1021-L1027
// TODO(buglloc): support kretprobe
func PerfEventOpenKprobe(funcName string, progFd int) (int, error) {
	typ, err := findPerfEventType(perfTypeKProbe)
	if err != nil {
		return 0, err
	}

	funcPtr := newStringPointer(funcName)
	attr := unix.PerfEventAttr{
		Type:   typ,
		Sample: 1,
		Wakeup: 1,
		Ext1:   uint64(uintptr(funcPtr)),
	}

	pfd, err := unix.PerfEventOpen(&attr, -1, 0, -1, unix.PERF_FLAG_FD_CLOEXEC)
	if err != nil {
		return -1, fmt.Errorf("unable to open perf events: %w", err)
	}

	if err := attachTracingEvent(pfd, progFd); err != nil {
		return -1, err
	}
	return pfd, nil
}

func attachTracingEvent(pfd, progFd int) error {
	if _, _, err := syscall.Syscall(syscall.SYS_IOCTL, uintptr(pfd), unix.PERF_EVENT_IOC_SET_BPF, uintptr(progFd)); err != 0 {
		return fmt.Errorf("error attaching bpf program to perf event: %w", err)
	}

	if _, _, err := syscall.Syscall(syscall.SYS_IOCTL, uintptr(pfd), unix.PERF_EVENT_IOC_ENABLE, 0); err != 0 {
		return fmt.Errorf("error enabling perf event: %w", err)
	}

	return nil
}

func newStringPointer(str string) unsafe.Pointer {
	// The kernel expects strings to be zero terminated
	buf := make([]byte, len(str)+1)
	copy(buf, str)

	return unsafe.Pointer(&buf[0])
}

func findPerfEventType(eventType string) (uint32, error) {
	if typ, ok := perfTypes.Load(eventType); ok {
		return typ.(uint32), nil
	}

	typePath := fmt.Sprintf(pmuTypeFile, eventType)
	rawType, err := ioutil.ReadFile(typePath)
	if err != nil {
		return 0, err
	}

	typ, err := strconv.ParseUint(string(bytes.TrimSpace(rawType)), 10, 32)
	if err != nil {
		return 0, fmt.Errorf("invalid pmu type (file=%s type=%q): %w", typePath, rawType, err)
	}

	out := uint32(typ)
	perfTypes.Store(eventType, out)
	return out, nil
}
