package podresolver

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strings"
	"sync"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/go-resty/resty/v2"
	"github.com/mailru/easyjson"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
)

const (
	issTimeout                = 2 * time.Second
	syncBackoffMaxInterval    = 5 * time.Second
	syncBackoffMaxElapsedTime = 2 * time.Minute
	syncMinInterval           = 1 * time.Second

	ypLiteDeployEngine = "AQ5ZUF9MSVRF"
	deployDeployEngine = "AQpNQ1JTQw=="
)

var ErrSlotNotFound = errors.New("slot was not found")

type Resolver struct {
	httpc   *resty.Client
	mu      sync.RWMutex
	pods    map[string]PodInfo
	doSync  chan struct{}
	sensors sensors.Sensor
	log     log.Logger
}

func NewResolver(opts ...Option) (*Resolver, error) {
	r := &Resolver{
		httpc: resty.New().
			SetBaseURL("http://localhost:25536/").
			SetTimeout(issTimeout),
		pods:    make(map[string]PodInfo),
		sensors: &sensors.NopSensor{},
		doSync:  make(chan struct{}, 1),
		log:     &nop.Logger{},
	}

	for _, opt := range opts {
		opt(r)
	}

	return r, nil
}

func (r *Resolver) ScheduleSync() {
	select {
	case r.doSync <- struct{}{}:
	default:
	}
}

func (r *Resolver) CachedSlot(containerName string) (PodInfo, error) {
	r.mu.RLock()
	defer r.mu.RUnlock()

	pod, ok := r.pods[containerName]
	if !ok {
		return ZeroPod, ErrSlotNotFound
	}
	return pod, nil
}

func (r *Resolver) Sync(ctx context.Context) bool {
	r.sensors.PodResolverSync(1)

	ok := true
	canDelete := true
	ypPods, err := r.ListYpPods(ctx)
	if err != nil {
		ok = false
		canDelete = false
		r.sensors.PodResolverErrors(1)
		r.log.Error("podresolver: failed to list YP Pods", log.Error(err))
	}

	issPods, err := r.ListNannyInstances(ctx)
	if err != nil {
		ok = false
		canDelete = false
		r.sensors.PodResolverErrors(1)
		r.log.Error("podresolver: failed to list Nanny instances", log.Error(err))
	}

	pods := append(ypPods, issPods...)

	var toDelete []string
	for c := range r.pods {
		found := false
		for i, p := range pods {
			if c == p.Container {
				lastItem := len(pods) - 1
				pods[i] = pods[lastItem]
				pods = pods[:lastItem]
				found = true
				break
			}
		}

		if !found {
			toDelete = append(toDelete, c)
		}
	}

	if len(pods) == 0 && (len(toDelete) == 0 || !canDelete) {
		// nothing changed
		return ok
	}

	r.mu.Lock()
	if canDelete {
		for _, c := range toDelete {
			delete(r.pods, c)
		}
	}

	for _, p := range pods {
		r.pods[p.Container] = p
	}
	r.mu.Unlock()
	return ok
}

func (r *Resolver) CacheWatcher(ctx context.Context) {
	syncErr := errors.New("sync fail")
	for {
		select {
		case <-ctx.Done():
			return
		case <-r.doSync:
			expBackOff := backoff.NewExponentialBackOff()
			expBackOff.MaxInterval = syncBackoffMaxInterval
			expBackOff.MaxElapsedTime = syncBackoffMaxElapsedTime
			expBackOff.Reset()

			err := backoff.RetryNotify(
				func() error {
					if !r.Sync(ctx) {
						return syncErr
					}
					return nil
				},
				backoff.WithContext(expBackOff, ctx),
				func(_ error, duration time.Duration) {
					r.log.Warn("podresolver: sync failed, wait next try", log.Duration("sleep", duration))
				},
			)

			switch err {
			case nil:
				// Sleep to prevent excessive re-resolutions
				time.Sleep(syncMinInterval)
			default:
				r.log.Error("podresolver: sync failed, wait next cgrps changes", log.Error(err))
			}
		}
	}
}
func (r *Resolver) ListYpPods(ctx context.Context) ([]PodInfo, error) {
	rsp, err := r.httpc.R().
		SetContext(ctx).
		SetDoNotParseResponse(true).
		Get("/pods/info")

	if err != nil {
		return nil, fmt.Errorf("failed to call ISS pods info: %w", err)
	}

	defer func() { _ = rsp.RawBody().Close() }()

	if rsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("non-200 ISS response: %s", rsp.Status())
	}

	var issPods ISSPods
	err = easyjson.UnmarshalFromReader(rsp.RawBody(), &issPods)
	if err != nil {
		return nil, fmt.Errorf("failed to unmarshal ISS pods info: %w", err)
	}

	out := make([]PodInfo, len(issPods.PodInfo))
	for i, pod := range issPods.PodInfo {
		kind := PodKindYp
		nannyID := pod.NannyService()
		if nannyID != "" {
			kind = PodKindNanny
		}

		out[i] = PodInfo{
			Kind:           kind,
			ID:             pod.ID,
			Container:      pod.ContainerName,
			PodSetID:       pod.PodSetID,
			NannyServiceID: nannyID,
		}
	}
	return out, nil
}

func (r *Resolver) ListNannyInstances(ctx context.Context) ([]PodInfo, error) {
	rsp, err := r.httpc.R().
		SetContext(ctx).
		SetDoNotParseResponse(true).
		Get("/instances")

	if err != nil {
		return nil, fmt.Errorf("failed to call ISS instances: %w", err)
	}

	defer func() { _ = rsp.RawBody().Close() }()

	if rsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("non-200 ISS response: %s", rsp.Status())
	}

	var issInstances ISSInstances
	err = easyjson.UnmarshalFromReader(rsp.RawBody(), &issInstances)
	if err != nil {
		return nil, fmt.Errorf("failed to unmarshal ISS instances: %w", err)
	}

	out := make([]PodInfo, 0, len(issInstances))
	for _, instance := range issInstances {
		if instance.InstanceData.DeployEngine != "" || instance.InstanceData.NannyServiceID == "" {
			continue
		}

		out = append(out, PodInfo{
			Kind:           PodKindNanny,
			ID:             SlotIDToPodID(instance.SlotID),
			Container:      instance.SlotContainer,
			NannyServiceID: instance.InstanceData.NannyServiceID,
		})
	}
	return out, nil
}

func SlotIDToPodID(slotID string) string {
	if idx := strings.IndexByte(slotID, '@'); idx > -1 {
		return slotID[0:idx]
	}

	return slotID
}
