package probe

import (
	"fmt"

	"gopkg.in/yaml.v2"
)

var (
	_ yaml.Marshaler   = (*Kind)(nil)
	_ yaml.Unmarshaler = (*Kind)(nil)
)

type Kind uint16

const (
	KindUnknown    Kind = 0
	KindNone       Kind = 1
	KindProcExec   Kind = 2
	KindSysExec    Kind = 3
	KindSysConnect Kind = 4
	KindSysPtrace  Kind = 5
	KindSysOpen    Kind = 6
)

func (pk Kind) MarshalYAML() (interface{}, error) {
	return pk.String(), nil
}

func (pk *Kind) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var kind string
	if err := unmarshal(&kind); err != nil {
		return err
	}

	switch kind {
	case "none", "":
		*pk = KindNone
	case "proc_exec":
		*pk = KindProcExec
	case "sys_exec":
		*pk = KindSysExec
	case "sys_connect":
		*pk = KindSysConnect
	case "sys_ptrace":
		*pk = KindSysPtrace
	case "sys_open":
		*pk = KindSysOpen
	default:
		return fmt.Errorf("unknown probe kind: %s", kind)
	}
	return nil
}

func (pk Kind) String() string {
	switch pk {
	case KindNone:
		return "none"
	case KindProcExec:
		return "proc_exec"
	case KindSysExec:
		return "sys_exec"
	case KindSysConnect:
		return "sys_connect"
	case KindSysPtrace:
		return "sys_ptrace"
	case KindSysOpen:
		return "sys_open"
	default:
		return "unknown"
	}
}
