package file

import (
	"context"
	"fmt"
	"strings"

	"gopkg.in/ini.v1"

	"a.yandex-team.ru/security/gideon/gideon/internal/secrets"
)

var _ secrets.Provider = (*Provider)(nil)

type Provider struct {
	cfg     secrets.Config
	secrets *ini.File
}

func NewProvider(cfg secrets.Config) (*Provider, error) {
	f, err := ini.Load(cfg.FilePath)
	if err != nil {
		return nil, fmt.Errorf("can't parse secret file %q: %w", cfg.FilePath, err)
	}

	return &Provider{
		cfg:     cfg,
		secrets: f,
	}, nil
}

func (p *Provider) GetSecret(_ context.Context, key string) (string, error) {
	iniKey, ok := p.cfg.Secrets[key]
	if !ok {
		iniKey = key
	}

	sectionName, keyName := parseIniKey(iniKey)
	section, err := p.secrets.GetSection(sectionName)
	if err != nil {
		return "", fmt.Errorf("section %s (%s) was not found in secrets file %q: %w", sectionName, key, p.cfg.FilePath, err)
	}

	out, err := section.GetKey(keyName)
	if err != nil {
		return "", fmt.Errorf("key %s (%s) was not found in secrets file %q: %w", keyName, key, p.cfg.FilePath, err)
	}

	return out.String(), nil
}

func (p *Provider) Close() error {
	return nil
}

func parseIniKey(key string) (string, string) {
	idx := strings.IndexByte(key, '.')
	if idx < 0 {
		return "", key
	}

	return key[:idx], key[idx+1:]
}
