package sensors

import (
	"sync/atomic"

	"a.yandex-team.ru/security/gideon/gideon/bpf"
)

/*
* documentation for stats semantic: https://wiki.yandex-team.ru/golovan/stat-handle/#defoltnyesigopt-suffiksy
 */
type UnistatSensor struct {
	eventUnknown     int64
	eventMkCgroup    int64
	eventRmCgroup    int64
	eventSyscall     int64
	eventProcExec    int64
	eventNewSession  int64
	bpfEventDrop     int64
	collectorBatch   int64
	collectorDrop    int64
	collectorError   int64
	collectorSize    int64
	cgrpMisses       int64
	slotMisses       int64
	podResolverSync  int64
	podResolverError int64
}

type PushValue struct {
	Name  string `json:"name"`
	Value int64  `json:"val"`
}

func NewUnistatSensors() *UnistatSensor {
	return &UnistatSensor{}
}

func (s *UnistatSensor) NewEvent(kind bpf.EventKind) {
	switch kind {
	case bpf.EventKindMkCgroup:
		atomic.AddInt64(&s.eventMkCgroup, 1)
	case bpf.EventKindRmCgroup:
		atomic.AddInt64(&s.eventRmCgroup, 1)
	case bpf.EventKindSyscall:
		atomic.AddInt64(&s.eventSyscall, 1)
	case bpf.EventKindProcExec:
		atomic.AddInt64(&s.eventProcExec, 1)
	case bpf.EventKindNewSession:
		atomic.AddInt64(&s.eventNewSession, 1)
	default:
		atomic.AddInt64(&s.eventUnknown, 1)
	}
}

func (s *UnistatSensor) BPFDropEvents(n int64) {
	atomic.AddInt64(&s.bpfEventDrop, n)
}

func (s *UnistatSensor) CollectorBatch(n int64) {
	atomic.AddInt64(&s.collectorBatch, n)
}

func (s *UnistatSensor) CollectorDrops(n int64) {
	atomic.AddInt64(&s.collectorDrop, n)
}

func (s *UnistatSensor) CollectorErrors(n int64) {
	atomic.AddInt64(&s.collectorError, n)
}

func (s *UnistatSensor) CollectorSize(n int64) {
	atomic.AddInt64(&s.collectorSize, n)
}

func (s *UnistatSensor) CgroupMiss(n int64) {
	atomic.AddInt64(&s.cgrpMisses, n)
}

func (s *UnistatSensor) SlotMiss(n int64) {
	atomic.AddInt64(&s.slotMisses, n)
}

func (s *UnistatSensor) PodResolverSync(n int64) {
	atomic.AddInt64(&s.podResolverSync, n)
}

func (s *UnistatSensor) PodResolverErrors(n int64) {
	atomic.AddInt64(&s.podResolverError, n)
}

func (s *UnistatSensor) FlushPush() []PushValue {
	return []PushValue{
		{"events_unknown_dmmm", atomic.SwapInt64(&s.eventUnknown, 0)},
		{"events_mkcgroup_dmmm", atomic.SwapInt64(&s.eventMkCgroup, 0)},
		{"events_rmcgroup_dmmm", atomic.SwapInt64(&s.eventRmCgroup, 0)},
		{"events_syscall_dmmm", atomic.SwapInt64(&s.eventSyscall, 0)},
		{"events_procexec_dmmm", atomic.SwapInt64(&s.eventProcExec, 0)},
		{"events_new_session_dmmm", atomic.SwapInt64(&s.eventNewSession, 0)},
		{"bpf_drops_dmmm", atomic.SwapInt64(&s.bpfEventDrop, 0)},
		{"collector_batches_dmmm", atomic.SwapInt64(&s.collectorBatch, 0)},
		{"collector_drops_dmmm", atomic.SwapInt64(&s.collectorDrop, 0)},
		{"collector_errors_dmmm", atomic.SwapInt64(&s.collectorError, 0)},
		{"collector_size_ahhh", atomic.LoadInt64(&s.collectorSize)},
		{"cgrp_misses_dmmm", atomic.SwapInt64(&s.cgrpMisses, 0)},
		{"slot_misses_dmmm", atomic.SwapInt64(&s.slotMisses, 0)},
		{"podresolver_syncs_dmmm", atomic.SwapInt64(&s.podResolverSync, 0)},
		{"podresolver_errors_dmmm", atomic.SwapInt64(&s.podResolverError, 0)},
	}
}
