package bob

import (
	"io"
	"unsafe"

	"a.yandex-team.ru/security/gideon/gideon/internal/byteorder"
)

type Reader struct {
	s []byte
	i int // current reading index
	m int // max len
}

func NewReader(in []byte) *Reader {
	return &Reader{
		s: in,
		i: 0,
		m: len(in),
	}
}

func (r *Reader) ReadUint8() (uint8, error) {
	if r.i+1 > r.m {
		return 0, io.EOF
	}

	out := r.s[r.i]
	r.i++
	return out, nil
}

func (r *Reader) ReadInt8() (int8, error) {
	if r.i+1 > r.m {
		return 0, io.EOF
	}

	out := r.s[r.i]
	r.i++
	return int8(out), nil
}

func (r *Reader) ReadUint16() (uint16, error) {
	if r.i+2 > r.m {
		return 0, io.EOF
	}

	out := byteorder.Host.Uint16(r.s[r.i:])
	r.i += 2
	return out, nil
}

func (r *Reader) ReadInt16() (int16, error) {
	if r.i+2 > r.m {
		return 0, io.EOF
	}

	out := byteorder.Host.Uint16(r.s[r.i:])
	r.i += 2
	return int16(out), nil
}

func (r *Reader) ReadUint32() (uint32, error) {
	if r.i+4 > r.m {
		return 0, io.EOF
	}

	out := byteorder.Host.Uint32(r.s[r.i:])
	r.i += 4
	return out, nil
}

func (r *Reader) ReadInt32() (int32, error) {
	if r.i+4 > r.m {
		return 0, io.EOF
	}

	out := byteorder.Host.Uint32(r.s[r.i:])
	r.i += 4
	return int32(out), nil
}

func (r *Reader) ReadUint64() (uint64, error) {
	if r.i+8 > r.m {
		return 0, io.EOF
	}

	out := byteorder.Host.Uint64(r.s[r.i:])
	r.i += 8
	return out, nil
}

func (r *Reader) ReadInt64() (int64, error) {
	if r.i+8 > r.m {
		return 0, io.EOF
	}

	out := byteorder.Host.Uint64(r.s[r.i:])
	r.i += 8
	return int64(out), nil
}

func (r *Reader) ReadBytes(size int) ([]byte, error) {
	if size == 0 {
		return nil, nil
	}

	if r.i+size > r.m {
		return nil, io.EOF
	}

	out := r.s[r.i : r.i+size]
	r.i += size
	return out, nil
}

func (r *Reader) ReadByteSlice() ([]byte, error) {
	rSize, err := r.ReadUint16()
	if err != nil {
		return nil, err
	}

	if rSize == 0 {
		return nil, nil
	}

	return r.ReadBytes(int(rSize))
}

func (r *Reader) ReadString() (string, error) {
	out, err := r.ReadByteSlice()
	if err != nil {
		return "", err
	}

	return *(*string)(unsafe.Pointer(&out)), nil
}

func (r *Reader) ReadStringCopy() (string, error) {
	out, err := r.ReadByteSlice()
	if err != nil {
		return "", err
	}

	return string(out), nil
}

func (r *Reader) Rest() ([]byte, error) {
	out := r.s[r.i:]
	r.i = r.m - 1
	return out, nil
}
