package tracer

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/gideon/bpf"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector"
	"a.yandex-team.ru/security/gideon/gideon/internal/podresolver"
	"a.yandex-team.ru/security/gideon/gideon/internal/probe"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
)

type Option func(t *Tracer) error

func WithDebug(debug bool) Option {
	return func(t *Tracer) error {
		t.debug = debug
		return nil
	}
}

func WithCollector(c collector.Collector) Option {
	return func(t *Tracer) error {
		t.collector = c
		return nil
	}
}

func WithLogger(l log.Logger) Option {
	return func(t *Tracer) error {
		t.log = l
		t.portos.log = l
		return nil
	}
}

func WithModulePath(modulePath string) Option {
	return func(t *Tracer) error {
		t.modulePath = modulePath
		return nil
	}
}

func WithMetrics(sensors sensors.Sensor) Option {
	return func(t *Tracer) error {
		t.sensors = sensors
		t.portos.sensors = sensors
		return nil
	}
}

func WithPodResolver(r *podresolver.Resolver) Option {
	return func(t *Tracer) error {
		t.portos.resolver = r
		return nil
	}
}

func WithRingBufferSize(size int) Option {
	return func(t *Tracer) error {
		t.ringBufferSize = size
		return nil
	}
}

func WithProbe(probes ...probe.Probe) Option {
	return func(t *Tracer) error {
		var (
			allProbes      = make(map[probe.Kind]struct{}, len(probes))
			rawProbes      = make(map[rawProgram]struct{})
			rawSyscalls    = make(map[bpf.SyscallKind]struct{})
			kprobeProbes   = make(map[kprobeProgram]struct{})
			kprobeSyscalls = make(map[bpf.SyscallKind]struct{})
			postFilters    = make(map[string]PostFilter)
		)

		requireRawProbe := func(name, tp string) {
			p := rawProgram{
				name:       name,
				tracepoint: tp,
			}
			rawProbes[p] = struct{}{}
		}

		requireRawSyscall := func(syscalls ...bpf.SyscallKind) {
			requireRawProbe("sys_exit", "sys_exit")
			for _, s := range syscalls {
				rawSyscalls[s] = struct{}{}
			}
		}

		requireKProbe := func(name, fn string) {
			p := kprobeProgram{
				name:     name,
				funcName: fn,
			}
			kprobeProbes[p] = struct{}{}
		}

		requireKprobeSyscall := func(syscalls ...bpf.SyscallKind) {
			requireKProbe("kprobe__seccomp_filter", "__seccomp_filter")
			for _, s := range syscalls {
				kprobeSyscalls[s] = struct{}{}
			}
		}

		requireKprobeSeccomp := func() {
			requireKProbe("kprobe_prctl_set_seccomp", "prctl_set_seccomp")
		}

		for _, p := range probes {
			if _, ok := allProbes[p.Kind]; ok {
				return fmt.Errorf("duplicate probe: %s", p.Kind)
			}

			allProbes[p.Kind] = struct{}{}
			switch p.Kind {
			case probe.KindProcExec:
				switch p.CollectKind {
				case probe.CollectKindAny:
					requireRawProbe("sched_process_exec", "sched_process_exec")
				case probe.CollectKindSlot:
					requireRawProbe("sched_process_exec_containers", "sched_process_exec")
					postFilters["post_filter"] = SlotPostFilter
				case probe.CollectKindSSH:
					requireKprobeSeccomp()
					requireRawProbe("sched_process_exec_seccomp", "sched_process_exec")
				default:
					return fmt.Errorf("probe %q doesn't support filtering kind %q", p.Kind, p.CollectKind)
				}
			case probe.KindSysExec:
				switch p.CollectKind {
				case probe.CollectKindAny:
					requireRawSyscall(bpf.SyscallKindExecVe, bpf.SyscallKindExecVeAt)
				case probe.CollectKindSSH:
					requireKprobeSeccomp()
					requireKprobeSyscall(bpf.SyscallKindExecVe, bpf.SyscallKindExecVeAt)
				default:
					return fmt.Errorf("probe %q doesn't support filtering kind %q", p.Kind, p.CollectKind)
				}
			case probe.KindSysPtrace:
				switch p.CollectKind {
				case probe.CollectKindAny:
					requireRawSyscall(bpf.SyscallKindPtrace)
				case probe.CollectKindSSH:
					requireKprobeSeccomp()
					requireKprobeSyscall(bpf.SyscallKindPtrace)
				default:
					return fmt.Errorf("probe %q doesn't support filtering kind %q", p.Kind, p.CollectKind)
				}
			case probe.KindSysConnect:
				switch p.CollectKind {
				case probe.CollectKindAny:
					requireRawSyscall(bpf.SyscallKindConnect)
				case probe.CollectKindSSH:
					requireKprobeSeccomp()
					requireKprobeSyscall(bpf.SyscallKindConnect)
				default:
					return fmt.Errorf("probe %q doesn't support filtering kind %q", p.Kind, p.CollectKind)
				}
			case probe.KindSysOpen:
				switch p.CollectKind {
				case probe.CollectKindAny:
					requireRawSyscall(bpf.SyscallKindOpen, bpf.SyscallKindOpenAt)
				case probe.CollectKindSSH:
					requireKprobeSeccomp()
					requireKprobeSyscall(bpf.SyscallKindOpen, bpf.SyscallKindOpenAt)
				default:
					return fmt.Errorf("probe %q doesn't support filtering kind %q", p.Kind, p.CollectKind)
				}
			default:
				return fmt.Errorf("unsupported probe: %s", p.Kind)
			}
		}

		for _, f := range postFilters {
			t.postFilters = append(t.postFilters, f)
		}

		requiredProbes := bpfProbes{
			rawProgs:       make([]rawProgram, 0, len(rawProbes)),
			rawSyscalls:    make([]bpf.SyscallKind, 0, len(rawSyscalls)),
			kprobeProgs:    make([]kprobeProgram, 0, len(kprobeProbes)),
			kprobeSyscalls: make([]bpf.SyscallKind, 0, len(kprobeSyscalls)),
		}

		for p := range rawProbes {
			requiredProbes.rawProgs = append(requiredProbes.rawProgs, p)
		}

		for s := range rawSyscalls {
			requiredProbes.rawSyscalls = append(requiredProbes.rawSyscalls, s)
		}

		for p := range kprobeProbes {
			requiredProbes.kprobeProgs = append(requiredProbes.kprobeProgs, p)
		}

		for s := range kprobeSyscalls {
			requiredProbes.kprobeSyscalls = append(requiredProbes.kprobeSyscalls, s)
		}

		t.bpfProbes = requiredProbes
		return nil
	}
}
