package tracer

import (
	"errors"
	"fmt"
	"net"
	"strings"
	"syscall"
	"time"
	"unsafe"

	"golang.org/x/sys/unix"

	"a.yandex-team.ru/security/gideon/gideon/bpf"
	"a.yandex-team.ru/security/gideon/gideon/internal/tracer/bob"
	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
)

//#include "gideon.h"
import "C"

var (
	bootTime = func() uint64 {
		var timespec syscall.Timespec
		_, _, err := syscall.Syscall(
			syscall.SYS_CLOCK_GETTIME,
			unix.CLOCK_MONOTONIC,
			uintptr(unsafe.Pointer(&timespec)),
			0,
		)

		if err != 0 {
			panic(fmt.Sprintf("syscall.SYS_CLOCK_GETTIME failed: %s", err.Error()))
		}

		return uint64(time.Now().UnixNano() - syscall.TimespecToNsec(timespec))
	}()

	zeroEventHeader = EventHeader{}

	errParseIgnore = errors.New("parse ignore")
)

type EventHeader struct {
	Kind bpf.EventKind
	TS   uint64
}

func (t *Tracer) readEventHeader(r *bob.Reader) (EventHeader, error) {
	data, err := r.ReadBytes(C.sizeof_struct_event_header_t)
	if err != nil {
		return zeroEventHeader, err
	}

	h := (*C.struct_event_header_t)(unsafe.Pointer(&data[0]))

	kind := bpf.EventKindFromUint(uint8(h.kind))
	if kind == bpf.EventKindUnknown {
		return zeroEventHeader, fmt.Errorf("unknown event kind: %d", h.kind)
	}

	return EventHeader{
		Kind: kind,
		TS:   bootTime + uint64(h.ts),
	}, nil
}

func (t *Tracer) readSyscall(r *bob.Reader, eventHeader EventHeader) (*events.Event, error) {
	procInfo, err := t.readProcInfo(r)
	if err != nil {
		return nil, fmt.Errorf("failed to read syscall procinfo: %w", err)
	}

	kind, err := r.ReadUint16()
	if err != nil {
		return nil, fmt.Errorf("failed to read syscall kind (pid=%d): %w", procInfo.Pid, err)
	}

	switch bpf.SyscallKindUint(kind) {
	case bpf.SyscallKindExecVeAt:
		event := events.ExecveAtEvent{}

		event.RetCode, err = r.ReadInt64()
		if err != nil {
			return nil, fmt.Errorf("failed to read execveat ret code (pid=%d): %w", procInfo.Pid, err)
		}

		event.Fd, err = r.ReadInt64()
		if err != nil {
			return nil, fmt.Errorf("failed to read execveat fd (pid=%d): %w", procInfo.Pid, err)
		}

		event.Filename, err = r.ReadString()
		if err != nil {
			return nil, fmt.Errorf("failed to read execveat filename (pid=%d): %w", procInfo.Pid, err)
		}

		args, err := r.ReadString()
		if err != nil {
			return nil, fmt.Errorf("failed to read execveat args (pid=%d): %w", procInfo.Pid, err)
		}

		if (len(args)) > 0 {
			event.Args = strings.Split(args, "\x00")
		}

		return &events.Event{
			Kind: events.EventKind_EK_SYS_EXECVE_AT,
			Ts:   eventHeader.TS,
			Proc: procInfo,
			Details: &events.Event_ExecveAt{
				ExecveAt: &event,
			},
		}, nil
	case bpf.SyscallKindOpenAt:
		event := events.OpenAtEvent{}

		event.RetCode, err = r.ReadInt64()
		if err != nil {
			return nil, fmt.Errorf("failed to read openat ret code (pid=%d): %w", procInfo.Pid, err)
		}

		event.Fd, err = r.ReadInt64()
		if err != nil {
			return nil, fmt.Errorf("failed to read openat fd (pid=%d): %w", procInfo.Pid, err)
		}

		event.Filename, err = r.ReadString()
		if err != nil {
			return nil, fmt.Errorf("failed to read openat filename (pid=%d): %w", procInfo.Pid, err)
		}

		event.Flags, err = r.ReadInt32()
		if err != nil {
			return nil, fmt.Errorf("failed to read openat flags (pid=%d): %w", procInfo.Pid, err)
		}

		return &events.Event{
			Kind: events.EventKind_EK_SYS_OPEN_AT,
			Ts:   eventHeader.TS,
			Proc: procInfo,
			Details: &events.Event_OpenAt{
				OpenAt: &event,
			},
		}, nil
	case bpf.SyscallKindPtrace:
		event := events.PtraceEvent{}

		event.RetCode, err = r.ReadInt64()
		if err != nil {
			return nil, fmt.Errorf("failed to read ptrace ret code (pid=%d): %w", procInfo.Pid, err)
		}

		event.Request, err = r.ReadInt64()
		if err != nil {
			return nil, fmt.Errorf("failed to read ptrace request (pid=%d): %w", procInfo.Pid, err)
		}

		event.Target, err = r.ReadInt32()
		if err != nil {
			return nil, fmt.Errorf("failed to read ptrace request (pid=%d): %w", procInfo.Pid, err)
		}

		return &events.Event{
			Kind: events.EventKind_EK_SYS_PTRACE,
			Ts:   eventHeader.TS,
			Proc: procInfo,
			Details: &events.Event_Ptrace{
				Ptrace: &event,
			},
		}, nil
	case bpf.SyscallKindConnect:
		rawFamily, err := r.ReadUint16()
		if err != nil {
			return nil, fmt.Errorf("failed to read connect family (pid=%d): %w", procInfo.Pid, err)
		}

		var (
			family  events.AddressFamilyKind
			srcAddr net.IP
			srcPort uint16
			dstAddr net.IP
			dstPort uint16
		)
		switch bpf.AddressFamilyKindUint(rawFamily) {
		case bpf.AfInet:
			family = events.AddressFamilyKind_AFK_INET
			srcAddr, err = r.ReadBytes(4)
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET saddr (pid=%d): %w", procInfo.Pid, err)
			}

			srcPort, err = r.ReadUint16()
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET sport (pid=%d): %w", procInfo.Pid, err)
			}

			dstAddr, err = r.ReadBytes(4)
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET daddr (pid=%d): %w", procInfo.Pid, err)
			}

			dstPort, err = r.ReadUint16()
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET dport (pid=%d): %w", procInfo.Pid, err)
			}
		case bpf.AfInet6:
			family = events.AddressFamilyKind_AFK_INET6
			srcAddr, err = r.ReadBytes(16)
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET6 saddr (pid=%d): %w", procInfo.Pid, err)
			}

			srcPort, err = r.ReadUint16()
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET6 sport (pid=%d): %w", procInfo.Pid, err)
			}

			dstAddr, err = r.ReadBytes(16)
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET6 daddr (pid=%d): %w", procInfo.Pid, err)
			}

			dstPort, err = r.ReadUint16()
			if err != nil {
				return nil, fmt.Errorf("failed to read AF_INET6 dport (pid=%d): %w", procInfo.Pid, err)
			}
		default:
			return nil, fmt.Errorf("incompatible address family (pid=%d): %d", procInfo.Pid, rawFamily)
		}

		return &events.Event{
			Kind: events.EventKind_EK_SYS_CONNECT,
			Ts:   eventHeader.TS,
			Proc: procInfo,
			Details: &events.Event_Connect{
				Connect: &events.ConnectEvent{
					Family:  family,
					SrcAddr: srcAddr.String(),
					SrcPort: uint32(srcPort),
					DstAddr: dstAddr.String(),
					DstPort: uint32(dstPort),
				},
			},
		}, nil
	default:
		return nil, fmt.Errorf("unknown syscall event (pid=%d): %d", procInfo.Pid, kind)
	}
}

func (t *Tracer) readProcExec(r *bob.Reader, eventHeader EventHeader) (*events.Event, error) {
	procInfo, err := t.readProcInfo(r)
	if err != nil {
		return nil, fmt.Errorf("failed to read procexec procinfo: %w", err)
	}

	event := events.ProcExecEvent{}
	event.Exe, err = r.ReadString()
	if err != nil {
		return nil, fmt.Errorf("failed to read procexec event exe: %w", err)
	}

	args, err := r.ReadString()
	if err != nil {
		return nil, fmt.Errorf("failed to read procexec args: %w", err)
	}

	if (len(args)) > 0 {
		event.Args = strings.Split(args, "\x00")
	}

	return &events.Event{
		Kind: events.EventKind_EK_PROC_EXEC,
		Ts:   eventHeader.TS,
		Proc: procInfo,
		Details: &events.Event_ProcExec{
			ProcExec: &event,
		},
	}, nil
}

func (t *Tracer) readNewSession(r *bob.Reader, eventHeader EventHeader) (*events.Event, error) {
	procInfo, err := t.readProcInfo(r)
	if err != nil {
		return nil, fmt.Errorf("failed to read procexec procinfo: %w", err)
	}

	if procInfo.SessionId == 0 {
		return nil, errors.New("no kernel session id")
	}

	procTitle, err := r.ReadString()
	if err != nil {
		return nil, fmt.Errorf("failed to read proctitle: %w", err)
	}

	env, err := r.ReadString()
	if err != nil {
		return nil, fmt.Errorf("failed to read env: %w", err)
	}

	tty, err := r.ReadString()
	if err != nil {
		return nil, fmt.Errorf("failed to read ttyname: %w", err)
	}

	sessKind, sessInfo := parseSessionInfo(procTitle, env)
	if sessKind == events.SessionKind_SK_UNSPECIFIED || len(sessInfo) == 0 {
		return nil, errParseIgnore
	}

	return &events.Event{
		Kind: events.EventKind_EK_SSH_SESSION,
		Ts:   eventHeader.TS,
		Proc: procInfo,
		Details: &events.Event_SshSession{
			SshSession: &events.SSHSessionEvent{
				Kind: sessKind,
				User: sessInfo["user"],
				Id:   sessInfo["session"],
				Tty:  tty,
			},
		},
	}, nil
}

func (t *Tracer) readProcInfo(r *bob.Reader) (*events.ProcInfo, error) {
	data, err := r.ReadBytes(C.sizeof_struct_proc_info_t)
	if err != nil {
		return nil, err
	}

	p := (*C.struct_proc_info_t)(unsafe.Pointer(&data[0]))
	protoc, _ := t.portos.CgroupIDToPod(uint64(p.cgid))

	comm := C.GoString((*C.char)(unsafe.Pointer(&p.comm)))
	parentComm := C.GoString((*C.char)(unsafe.Pointer(&p.parent_comm)))

	return &events.ProcInfo{
		Pid:        uint32(p.pid & 0xffffffff),
		Name:       comm,
		Ppid:       uint32(p.ppid & 0xffffffff),
		ParentName: parentComm,
		Uid:        uint32(p.uid & 0xffffffff),
		Container:  protoc.Container,
		PodId:      protoc.PodID,
		PodSetId:   protoc.PodSetID,
		NannyId:    protoc.NannyID,
		SessionId:  uint32(p.sessid & 0xffffffff),
	}, nil
}

func parseSessionLeaderTitle(in string) map[string]string {
	out := make(map[string]string)
	s := in[len(sessionLeaderHeader):]
	for {
		l := strings.IndexByte(s, '[')
		if l < 0 {
			break
		}

		r := strings.IndexByte(s[l:], ']')
		if r <= 0 {
			break
		}

		z := strings.SplitN(s[l+1:l+r], "=", 2)
		if len(z) < 2 {
			break
		}

		out[z[0]] = z[1]

		s = s[r:]
	}
	return out
}

func parseYtJobShellEnv(rawEnv string) map[string]string {
	env := strings.Split(rawEnv, "\x00")
	out := make(map[string]string)
	for _, e := range env {
		if !strings.HasPrefix(e, "YT_SHELL_ID=") {
			continue
		}
		out["session"] = e[12:]
	}
	return out
}

func parseSessionInfo(proctitle, env string) (events.SessionKind, map[string]string) {
	if strings.HasPrefix(proctitle, sessionLeaderHeader) {
		return events.SessionKind_SK_PORTOSHELL, parseSessionLeaderTitle(proctitle)
	}

	if strings.Contains(env, "YT_SHELL_ID=") {
		return events.SessionKind_SK_YT_JOBSHELL, parseYtJobShellEnv(env)
	}

	return 0, nil
}
