package tracer

import (
	"strings"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/gideon/internal/podresolver"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
)

type Pod struct {
	Container string
	PodID     string
	PodSetID  string
	NannyID   string
}

type portoCgroup struct {
	slot string
	path string
}

type PodStorage struct {
	cgroups   map[uint64]portoCgroup
	cgroupsMu sync.RWMutex
	log       log.Logger
	sensors   sensors.Sensor
	resolver  *podresolver.Resolver
}

func (ps *PodStorage) CgroupIDToPod(cgrID uint64) (Pod, bool) {
	ps.cgroupsMu.RLock()
	cgroup, ok := ps.cgroups[cgrID]
	ps.cgroupsMu.RUnlock()

	if !ok {
		ps.sensors.CgroupMiss(1)
		return Pod{}, false
	}

	out := Pod{
		Container: cgroup.path,
	}

	slotName := cgroup.slot
	if slotName == "" || ps.resolver == nil {
		return out, true
	}

	slotInfo, err := ps.resolver.CachedSlot(slotName)
	if err != nil {
		ps.sensors.SlotMiss(1)
		return out, true
	}

	out.PodID = slotInfo.ID
	out.PodSetID = slotInfo.PodSetID
	out.NannyID = slotInfo.NannyServiceID
	return out, true
}

func (ps *PodStorage) storeCgroup(cgrID uint64, cgrp portoCgroup) {
	if cgrID == 0 {
		return
	}

	ps.cgroupsMu.Lock()
	defer ps.cgroupsMu.Unlock()
	ps.cgroups[cgrID] = cgrp
}

func (ps *PodStorage) deleteCgroup(cgrID uint64) {
	if cgrID == 0 {
		return
	}

	ps.cgroupsMu.Lock()
	defer ps.cgroupsMu.Unlock()
	delete(ps.cgroups, cgrID)
}

func (ps *PodStorage) OnMkCgroup(cfgID uint64, path string) {
	if !isPortoCgroup(path) {
		ps.log.Debug("ignore non-porto cgroup", log.UInt64("id", cfgID), log.String("path", path))
		return
	}

	if ps.resolver != nil && isSlotCgroup(path) {
		ps.resolver.ScheduleSync()
	}

	slot := slotContainer(path)
	ps.storeCgroup(cfgID, portoCgroup{
		path: path,
		slot: slot,
	})
}

func (ps *PodStorage) OnRmCgroup(cfgID uint64, path string) {
	if !isPortoCgroup(path) {
		ps.log.Debug("ignore non-porto cgroup", log.UInt64("id", cfgID), log.String("path", path))
		return
	}

	if ps.resolver != nil && isSlotCgroup(path) {
		ps.resolver.ScheduleSync()
	}

	ps.deleteCgroup(cfgID)
}

func isPortoCgroup(path string) bool {
	return strings.HasPrefix(path, portoRootPath)
}

func isSlotCgroup(path string) bool {
	return strings.HasPrefix(path, slotContainerPrefix) && strings.Count(path, "/") == 2
}

func slotContainer(cgroupPath string) string {
	if cgroupPath == "" || !strings.HasPrefix(cgroupPath, slotContainerPrefix) {
		return ""
	}

	result := cgroupPath[len(portoRootPath):]
	idx := strings.IndexByte(result, '/')
	if idx <= 0 {
		return result
	}

	return result[:idx]
}
