package tracer

import (
	"context"
	"fmt"
	"io/fs"
	"io/ioutil"
	"path/filepath"
	"strconv"
	"strings"
	"time"

	"github.com/cilium/ebpf"
	"github.com/cilium/ebpf/link"
	"github.com/davecgh/go-spew/spew"
	"github.com/gofrs/uuid"
	"golang.org/x/sys/unix"

	"a.yandex-team.ru/library/go/core/log"
	nopLog "a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/gideon/gideon/bpf"
	"a.yandex-team.ru/security/gideon/gideon/internal/cgroup"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector"
	"a.yandex-team.ru/security/gideon/gideon/internal/collector/stdcollector"
	"a.yandex-team.ru/security/gideon/gideon/internal/ebpfutil"
	"a.yandex-team.ru/security/gideon/gideon/internal/kernel"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
	"a.yandex-team.ru/security/gideon/gideon/internal/tracer/bob"
	"a.yandex-team.ru/security/gideon/gideon/internal/tracer/perf"
	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
)

type (
	Tracer struct {
		log            log.Logger
		ringBufferSize int
		debug          bool
		uuidGen        uuid.Generator
		bpfCollection  *ebpf.Collection
		collector      collector.Collector
		sensors        sensors.Sensor
		consumeDone    chan struct{}
		eventsReader   *perf.Reader
		modulePath     string
		portos         PodStorage
		fds            []int
		links          []link.Link
		bpfProbes      bpfProbes
		postFilters    []PostFilter
	}

	bpfProbes struct {
		rawProgs       []rawProgram
		rawSyscalls    []bpf.SyscallKind
		kprobeProgs    []kprobeProgram
		kprobeSyscalls []bpf.SyscallKind
		constants      map[string]interface{}
	}
)

func NewTracer(opts ...Option) (*Tracer, error) {
	t := &Tracer{
		sensors:        &sensors.NopSensor{},
		ringBufferSize: bpfBufferSize,
		debug:          false,
		consumeDone:    make(chan struct{}),
		log:            &nopLog.Logger{},
		uuidGen:        uuid.NewGen(),
		portos: PodStorage{
			log:     &nopLog.Logger{},
			sensors: &sensors.NopSensor{},
			cgroups: make(map[uint64]portoCgroup),
		},
	}

	for _, opt := range opts {
		if err := opt(t); err != nil {
			return nil, fmt.Errorf("failed to apply option: %w", err)
		}
	}

	if t.collector == nil {
		t.collector = stdcollector.NewCollector(t.log)
	}
	return t, nil
}

func (t *Tracer) Close(ctx context.Context) {
	if t.eventsReader != nil {
		// stop perf maps
		err := t.eventsReader.Close()
		if err != nil {
			t.log.Error("failed to stop events reader", log.Error(err))
		}

		// grateful wait consumes
		select {
		case <-t.consumeDone:
			// completed normally
		case <-ctx.Done():
			// timed out
			return
		}
	}

	for _, l := range t.links {
		if err := l.Close(); err != nil {
			t.log.Error("failed to close BPF Link")
		}
	}

	if t.bpfCollection != nil {
		t.bpfCollection.Close()
	}

	for _, fd := range t.fds {
		if fd == -1 {
			continue
		}

		if err := unix.Close(fd); err != nil {
			t.log.Error("failed to close FD", log.Int("fd", fd), log.Error(err))
		}
	}
}

func (t *Tracer) InitBPF() error {
	kernelVer, err := kernel.CurrentVersion()
	if err != nil {
		return fmt.Errorf("failed to get current kernel version: %w", err)
	}

	var spec *ebpf.CollectionSpec
	if t.modulePath == "" {
		var key string
		spec, key, err = bpf.NewCollectionSpec(kernelVer)
		t.log.Info("loaded internal bpf program", log.String("resource", key))
	} else {
		spec, err = ebpf.LoadCollectionSpec(t.modulePath)
		t.log.Info("loaded external bpf program", log.String("path", t.modulePath))
	}

	if err != nil {
		return err
	}

	settings, err := t.bpfSettings(spec.Maps[settingsMapName])
	if err != nil {
		return fmt.Errorf("can't create settings map: %w", err)
	}

	err = spec.RewriteMaps(map[string]*ebpf.Map{
		settingsMapName: settings,
	})
	if err != nil {
		return fmt.Errorf("can't rewrite settings map: %w", err)
	}

	ebpfutil.RemoveUnusedProgs(spec, t.usedPrograms()...)
	ebpfutil.FixUpCollectionSpec(spec)
	t.bpfCollection, err = ebpf.NewCollectionWithOptions(spec, ebpf.CollectionOptions{
		Programs: ebpf.ProgramOptions{
			LogSize: bpfLogSize,
		},
	})
	return err
}

func (t *Tracer) InitConsumer() error {
	// init perf map
	perfMap, ok := t.bpfCollection.Maps[eventMapName]
	if !ok {
		return fmt.Errorf("can't find events map: %s", eventMapName)
	}

	var err error
	t.eventsReader, err = perf.NewReader(perfMap, t.ringBufferSize)
	if err != nil {
		return fmt.Errorf("can't initialize events reader: %w", err)
	}
	return nil
}

func (t *Tracer) Consume() error {
	defer close(t.consumeDone)

	parseEvent := func(data []byte) {
		reader := bob.NewReader(data)
		eventHeader, err := t.readEventHeader(reader)
		if err != nil {
			t.log.Error("failed to read input event kind", log.Error(err))
			return
		}

		var event *events.Event
		switch eventHeader.Kind {
		case bpf.EventKindSyscall:
			event, err = t.readSyscall(reader, eventHeader)
			if err != nil {
				t.log.Error("failed to read syscall event", log.Error(err))
				return
			}
		case bpf.EventKindProcExec:
			event, err = t.readProcExec(reader, eventHeader)
			if err != nil {
				t.log.Error("failed to read procexec event", log.Error(err))
				return
			}
		case bpf.EventKindNewSession:
			event, err = t.readNewSession(reader, eventHeader)
			if err != nil {
				if err != errParseIgnore {
					t.log.Error("failed to read new_session event", log.Error(err))
				}
				return
			}
		case bpf.EventKindMkCgroup:
			id, err := reader.ReadUint64()
			if err != nil {
				t.log.Error("failed to read new cgroup id", log.Error(err))
				return
			}

			name, err := reader.ReadStringCopy()
			if err != nil {
				t.log.Error("failed to read new cgroup path", log.Error(err))
				return
			}

			t.portos.OnMkCgroup(id, name)
			t.log.Debug("new container", log.UInt64("id", id), log.String("path", name))
			return
		case bpf.EventKindRmCgroup:
			id, err := reader.ReadUint64()
			if err != nil {
				t.log.Error("failed to read rm cgroup id", log.Error(err))
				return
			}

			name, err := reader.ReadStringCopy()
			if err != nil {
				t.log.Error("failed to read rm cgroup path", log.Error(err))
				return
			}

			t.portos.OnRmCgroup(id, name)
			t.log.Debug("container destroyed", log.UInt64("id", id), log.String("path", name))
			return
		default:
			t.log.Error("unknown event kind",
				log.UInt16("event_kind", uint16(eventHeader.Kind)),
				log.UInt64("ts", eventHeader.TS))
			return
		}

		for _, f := range t.postFilters {
			if !f(event) {
				return
			}
		}

		t.sensors.NewEvent(eventHeader.Kind)
		t.collector.NewEvent(event)
	}

	for {
		record, err := t.eventsReader.Read()
		if err != nil {
			if perf.IsClosed(err) {
				// ok
				break
			}

			t.log.Error("could not read event from eBPF", log.Error(err))
			continue
		}

		if record.LostSamples > 0 {
			t.sensors.BPFDropEvents(int64(record.LostSamples))
			continue
		}

		if t.debug {
			fmt.Printf("new event (%s) from CPU %d: \n", time.Now(), record.CPU)
			spew.Dump(record.RawSample)
		}

		if len(record.RawSample) == 0 {
			// huh
			continue
		}

		parseEvent(record.RawSample)
		record.Release()
	}

	return nil
}

func (t *Tracer) Restore() error {
	addCgroup := func(id uint64, path string) {
		name := strings.TrimPrefix(path, freezerCgroupPath)
		if name == "" {
			name = "/"
		}

		t.log.Info("found existed freezer cgroup", log.UInt64("id", id), log.String("path", name))
		t.portos.OnMkCgroup(id, name)
	}

	err := filepath.WalkDir(freezerCgroupPath, func(osPathname string, d fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if !d.IsDir() {
			return nil
		}

		id, err := cgroup.GetCgroupID(osPathname)
		if err != nil {
			return fmt.Errorf("get freezer cgroup %q id fail: %w", osPathname, err)
		}

		addCgroup(id, osPathname)
		return nil
	})

	if err != nil {
		return fmt.Errorf("freezer cgroup tree walk fail: %w", err)
	}

	return nil
}

func (t *Tracer) EnablePortoProbes() error {
	for _, tp := range portoTracepoints {
		prog, ok := t.bpfCollection.Programs[tp]
		if !ok {
			return fmt.Errorf("failed to find bpf program, required for porto probes: %q", tp)
		}

		l, err := link.AttachRawTracepoint(link.RawTracepointOptions{
			Name:    tp,
			Program: prog,
		})

		if err != nil {
			return fmt.Errorf("failed to attach to raw tracepoint(%q), required for porto probes: %w", tp, err)
		}

		t.links = append(t.links, l)
	}

	return nil
}

func (t *Tracer) EnableProbes() error {
	if err := t.attachRawSyscall(t.bpfProbes.rawSyscalls...); err != nil {
		return err
	}

	if err := t.attachSeccompSyscall(t.bpfProbes.kprobeSyscalls...); err != nil {
		return err
	}

	if err := t.attachRawBPF(t.bpfProbes.rawProgs...); err != nil {
		return err
	}

	if err := t.attachKprobeBPF(t.bpfProbes.kprobeProgs...); err != nil {
		return err
	}

	return nil
}

func (t *Tracer) EmitNewSession(pid uint32, user, tty string) (string, error) {
	sessionID, err := readFileUInt(fmt.Sprintf(sessionIDFmt, pid))
	if err != nil {
		return "", fmt.Errorf("can't read sessionid: %w", err)
	}

	extSessionUUID, err := t.uuidGen.NewV4()
	if err != nil {
		return "", fmt.Errorf("can't generate sessionid: %w", err)
	}
	extSessionID := extSessionUUID.String()

	t.sensors.NewEvent(bpf.EventKindNewSession)

	event := &events.Event{
		Kind: events.EventKind_EK_SSH_SESSION,
		Ts:   uint64(time.Now().UnixNano()),
		Proc: &events.ProcInfo{
			Pid:       pid,
			SessionId: uint32(sessionID),
		},
		Details: &events.Event_SshSession{
			SshSession: &events.SSHSessionEvent{
				Kind: events.SessionKind_SK_SSH,
				Id:   extSessionID,
				User: user,
				Tty:  tty,
			},
		},
	}

	t.collector.NewEvent(event)
	return extSessionID, nil
}

func readFileUInt(p string) (uint64, error) {
	b, err := ioutil.ReadFile(p)
	if err != nil {
		return 0, err
	}

	return strconv.ParseUint(string(b), 10, 64)
}
