package yasm

import (
	"context"
	"fmt"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/gideon/gideon/internal/sensors"
)

const (
	DefaultUpstream = "http://localhost:11005"
	DefaultTimeout  = 100 * time.Millisecond
)

type Pusher struct {
	httpc   *resty.Client
	metrics *sensors.UnistatSensor
	log     log.Logger
	data    []pushData
}

type pushData struct {
	Tags   map[string]string   `json:"tags,omitempty"`
	TTL    uint32              `json:"ttl"`
	Values []sensors.PushValue `json:"values"`
}

func NewPusher(metrics *sensors.UnistatSensor, opts ...Option) *Pusher {
	httpc := resty.New().
		SetHeaders(map[string]string{
			"Content-Type": "application/json",
			"User-Agent":   "Gideon YASM pusher",
		}).
		SetBaseURL(DefaultUpstream).
		SetTimeout(DefaultTimeout)

	p := &Pusher{
		httpc:   httpc,
		metrics: metrics,
		log:     &nop.Logger{},
		data:    []pushData{{}},
	}

	for _, opt := range opts {
		opt(p)
	}

	return p
}

func (p *Pusher) Push() error {
	p.data[0].Values = p.metrics.FlushPush()

	rsp, err := p.httpc.R().
		SetBody(p.data).
		Post("/")
	if err != nil {
		return err
	}

	if !rsp.IsSuccess() {
		return fmt.Errorf("non-200 response: %d", rsp.StatusCode())
	}
	return nil
}

func (p *Pusher) Start(ctx context.Context, interval time.Duration) {
	var err error
	for {
		toNextPush := time.Until(
			time.Now().Add(interval).Truncate(interval),
		)

		select {
		case <-ctx.Done():
			return
		case <-time.After(toNextPush):
			err = p.Push()
			if err != nil {
				p.log.Warn("push metrics fail", log.Error(err))
			}
		}
	}
}
