// Package protoseq implements a coding scheme used in LogFeller in order
// collect separate Protobuf messages into one frame.
//
// See https://wiki.yandex-team.ru/logfeller/splitter/protoseq/ for details.
package protoseq

import (
	"bytes"
	"encoding/binary"
	"errors"
	"io"
)

var (
	ErrInInitialState = errors.New(".More() should be called first")
	ErrWrongSyncWord  = errors.New("wrong sync word")
)

var (
	protoseqMagic = [32]byte{
		0x1F, 0xF7, 0xF7, 0x7E, 0xBE, 0xA6, 0x5E, 0x9E, 0x37, 0xA6, 0xF6, 0x2E, 0xFE, 0xAE, 0x47, 0xA7, 0xB7, 0x6E,
		0xBF, 0xAF, 0x16, 0x9E, 0x9F, 0x37, 0xF6, 0x57, 0xF7, 0x66, 0xA7, 0x06, 0xAF, 0xF7,
	}
)

type decoderState byte

const (
	initState decoderState = iota
	moreState
	decodeState
	terminalState
)

type Message interface {
	Unmarshal(data []byte) error
}

// Decoder decodes a Protoseq frame into a batch of type-identical Protobuf
// messages.
type Decoder struct {
	r       io.Reader
	err     error
	buf     [32]byte     // Length of payload.
	payload bytes.Buffer // Protobuf payload itself.
	s       decoderState
}

func NewDecoder(r io.Reader) *Decoder {
	return &Decoder{r: r, s: initState}
}

func (d *Decoder) Reset(r io.Reader) {
	d.r = r
	d.s = initState
	d.payload.Reset()
	d.err = nil
}

func (d *Decoder) Decode(v Message) error {
	switch d.s {
	case initState:
		d.transitToTerminalState(ErrInInitialState)
		return d.err
	case terminalState:
		return d.err // In terminal state already.
	case moreState:
		d.s = decodeState
		fallthrough
	case decodeState:
		return v.Unmarshal(d.payload.Bytes())
	default:
		panic("unreachable state")
	}
}

func (d *Decoder) Err() error {
	return d.err
}

func (d *Decoder) More() bool {
	switch d.s {
	case moreState:
		return true // Already in moreState.
	case terminalState:
		return false
	case initState:
		fallthrough
	case decodeState:
		if _, err := io.ReadFull(d.r, d.buf[:4]); err != nil {
			if err == io.EOF {
				// this is fine, just end of stream
				err = nil
			}

			d.transitToTerminalState(err)
			return false
		}

		d.payload.Reset()
		size := binary.LittleEndian.Uint32(d.buf[:4])
		if _, err := io.CopyN(&d.payload, d.r, int64(size)); err != nil {
			d.transitToTerminalState(err)
			return false
		}

		if _, err := io.ReadFull(d.r, d.buf[:]); err != nil {
			d.transitToTerminalState(err)
			return false
		}

		if protoseqMagic != d.buf {
			d.transitToTerminalState(ErrWrongSyncWord)
			return false
		}

		d.s = moreState
		return true
	default:
		panic("unreachable state")
	}
}

func (d *Decoder) transitToTerminalState(err error) {
	d.s = terminalState
	d.err = err
}
