package commands

import (
	"context"
	"errors"
	"fmt"
	"regexp"
	"strings"
	"time"

	"github.com/spf13/cobra"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/encoding/prototext"

	"a.yandex-team.ru/security/gideon/nuvault/pkg/nuvault"
	"a.yandex-team.ru/security/gideon/nuvault/pkg/nuvrpc"
)

var secretFlags = struct {
	CertPath string
	Format   string
	Retries  int
	Timeout  time.Duration
}{
	Format:  "text",
	Retries: nuvault.DefaultRetries,
	Timeout: nuvault.DefaultTimeout,
}

var secretCmd = &cobra.Command{
	Use:   "secret [flags] secretUUID_0 secretUUID_N",
	Short: "get secret from NuVault",
	RunE: func(_ *cobra.Command, args []string) error {
		if len(args) == 0 {
			return errors.New("no secrets provided")
		}

		var printer func(*nuvrpc.Secret) error
		switch secretFlags.Format {
		case "json":
			printer = func(s *nuvrpc.Secret) error {
				out, err := protojson.Marshal(s)
				if err != nil {
					return err
				}

				fmt.Println(string(out))
				return nil
			}
		case "text":
			printer = func(s *nuvrpc.Secret) error {
				fmt.Println("----------")
				out, err := prototext.Marshal(s)
				if err != nil {
					return err
				}

				fmt.Println(string(out))
				return nil
			}
		case "env":
			/*
				Environment variable names used by the utilities in the Shell and Utilities volume of IEEE Std 1003.1-2001 consist solely of uppercase letters, digits, and the '_' (underscore)
				https://pubs.opengroup.org/onlinepubs/000095399/basedefs/xbd_chap08.html
			*/
			shellQuote := func(in string) string {
				return fmt.Sprintf("'%s'", strings.ReplaceAll(in, "'", "'\"'\"'"))
			}

			envKeyRe := regexp.MustCompile("^[a-zA-Z0-9_]+$")
			printer = func(s *nuvrpc.Secret) error {
				var out strings.Builder
				for k, v := range s.Values {
					if !envKeyRe.MatchString(k) {
						return fmt.Errorf("invalid secret key: must be %q: %s", envKeyRe.String(), k)
					}

					envKey := strings.ToUpper(k)
					out.WriteString(envKey)
					out.WriteByte('=')
					out.WriteString(shellQuote(v))
					out.WriteString("; export ")
					out.WriteString(envKey)
					out.WriteString(";\n")
				}

				fmt.Println(out.String())
				return nil
			}
		default:
			return fmt.Errorf("unsupported output format (supported: json,text,env): %s", secretFlags.Format)
		}

		client, err := nuvault.NewClient(
			nuvault.WithAddr(apiAddr),
			nuvault.WithMutualAuth(secretFlags.CertPath),
			nuvault.WithMaxRetries(secretFlags.Retries),
			nuvault.WithTimeout(secretFlags.Timeout),
		)
		if err != nil {
			return err
		}

		defer func() {
			_ = client.Close()
		}()

		for _, secretUUID := range args {
			secret, err := client.GetSecret(context.Background(), secretUUID)
			if err != nil {
				return fmt.Errorf("failed to get secret %s: %w", secretUUID, err)
			}

			if err := printer(secret); err != nil {
				return fmt.Errorf("failed to print secret %s: %w", secretUUID, err)
			}
		}
		return nil
	},
}

func init() {
	flags := secretCmd.PersistentFlags()
	flags.StringVar(&secretFlags.CertPath, "crt", "/etc/certs/capi.pem", "client crt path")
	flags.StringVar(&secretFlags.Format, "format", secretFlags.Format, "output format (json, text, env)")
	flags.IntVar(&secretFlags.Retries, "retries", secretFlags.Retries, "maximum number of retries")
	flags.DurationVar(&secretFlags.Timeout, "timeout", secretFlags.Timeout, "timeout to get one secret info")

	rootCmd.AddCommand(secretCmd)
}
